package com.xebialabs.deployit.deployment.planner;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.api.reflect.ReflectionsHolder.getMethodsAnnotatedWith;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Set;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.planning.Noop;
import com.xebialabs.deployit.plugin.api.deployment.planning.PostPlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.execution.Step;

public class DeploymentPlannerFactory {
	
    public DeploymentPlanner planner() {

        ListMultimap<Operation, Method> deployedContributors = registerDeployedContributors();
        Set<Method> contributors = checkContributors(getMethodsAnnotatedWith(Contributor.class));
	    List<Method> preProcessors = newArrayList(checkProcessors(getMethodsAnnotatedWith(PrePlanProcessor.class)));
	    List<Method> postProcessors = newArrayList(checkProcessors(getMethodsAnnotatedWith(PostPlanProcessor.class)));

        return new DeploymentPlanner.DeploymentPlannerBuilder()
                .preProcessors(preProcessors)
                .postProcessors(postProcessors)
                .typeContributors(deployedContributors)
                .contributors(contributors)
                .build();
    }

    private Set<Method> checkProcessors(Set<Method> processors) {
        for (Method processor : processors) {
            checkArgument(processor.getReturnType().equals(Step.class) || processor.getReturnType().equals(List.class), "Pre/Post processor %s should have a Step or List<Step> return type.", processor);
            Class<?>[] parameterTypes = processor.getParameterTypes();
            checkArgument(parameterTypes.length == 1, "Processor %s should take 1 parameter.", processor);
            checkArgument(parameterTypes[0].equals(DeltaSpecification.class), "Processor %s should take %s as first parameter.", processor, DeltaSpecification.class);
        }
        return processors;
    }

    private Set<Method> checkDeployedContributors(Set<Method> deployedContributors) {
        for (Method c : deployedContributors) {
            checkArgument(c.getReturnType().equals(void.class), "DeployedContributor %s should have void return type.", c);
            Class<?>[] parameterTypes = c.getParameterTypes();
            checkArgument(parameterTypes.length <= 2 && parameterTypes.length >= 1, "DeployedContributor %s should take 1 or 2 parameters.", c);
            checkArgument(parameterTypes[0].equals(DeploymentPlanningContext.class), "DeployedContributor %s should take %s as first parameter.", c, DeploymentPlanningContext.class);
            if (parameterTypes.length == 2) {
                checkArgument(parameterTypes[1].equals(Delta.class), "DeployedContributor %s should take %s as first parameter.", c, Delta.class);
            }
        }
        return deployedContributors;
    }

    private Set<Method> checkContributors(Set<Method> contributors) {
        for (Method contributor : contributors) {
            checkArgument(contributor.getReturnType().equals(void.class), "Contributor %s should have void return type.", contributor);
            Class<?>[] parameterTypes = contributor.getParameterTypes();
            checkArgument(parameterTypes.length == 2, "Contributor %s should take 2 parameters.", contributor);
            checkArgument(parameterTypes[0].equals(Deltas.class), "Contributor %s should take %s as first parameter.", contributor, Deltas.class);
            checkArgument(parameterTypes[1].equals(DeploymentPlanningContext.class), "Contributor %s should take %s as second parameter.", contributor, DeploymentPlanningContext.class);
        }
        return contributors;
    }

	private ListMultimap<Operation, Method> registerDeployedContributors() {
        ListMultimap<Operation, Method> deployedContributorMap = ArrayListMultimap.create();

	    registerDeployedContributors(Create.class, Operation.CREATE, deployedContributorMap);
	    registerDeployedContributors(Modify.class, Operation.MODIFY, deployedContributorMap);
	    registerDeployedContributors(Destroy.class, Operation.DESTROY, deployedContributorMap);
	    registerDeployedContributors(Noop.class, Operation.NOOP, deployedContributorMap);

        return deployedContributorMap;
    }

	private void registerDeployedContributors(Class<? extends Annotation> annotation, Operation operation, ListMultimap<Operation, Method> deployedContributorMap) {
		Set<Method> typeContributors = checkDeployedContributors(getMethodsAnnotatedWith(annotation));
		for (Method typeContributor : typeContributors) {
			deployedContributorMap.put(operation, typeContributor);
		}
	}
}
