/*
 * This file is part of Maven Deployit plugin.
 *
 * Maven Deployit plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Maven Deployit plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maven Deployit plugin.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xebialabs.deployit.maven;

import java.util.List;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.engine.api.DeploymentService;
import com.xebialabs.deployit.engine.api.RepositoryService;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.maven.helper.DeploymentHelper;
import com.xebialabs.deployit.maven.logic.CustomDeployedsApplicationStrategy;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;

import static java.lang.String.format;

/**
 * Deploy artifacts to the target environment.
 *
 * @author Benoit Moussaud
 * @goal deploy
 * @phase pre-integration-test
 * @requiresDependencyResolution compile
 * @configurator override
 */

public class DeployMojo extends AbstractDeployitMojo {

    private String currentVersion;
    private Deployment generatedDeployment;

    public void execute() throws MojoExecutionException, MojoFailureException {

        getLog().info("deployit:deploy");
        boot();
        try {
            deploy();
        } finally {
            shutdown();
        }
        getLog().info("End of deploy:deploy");
    }

    private void deploy() throws MojoExecutionException {
        final ConfigurationItem deploymentPackage = uploadPackage();
        final ConfigurationItem targetEnvironment = getTargetEnvironment();

        Boolean update = getDeploymentHelper().isApplicationDeployed(deploymentPackage.getId(), targetEnvironment.getId());
        generateDeployment(deploymentPackage, targetEnvironment, update);

        getLog().info("Deployeds to be included into generatedDeployment:");
        for (ConfigurationItem d : generatedDeployment.getDeployeds()) {
            getLog().info("    -> " + d.getId());
        }

        if (!Strings.isNullOrEmpty(orchestrator)) {
            getLog().info("Using orchestrator: " + orchestrator);
            generatedDeployment.getDeployedApplication().setProperty("orchestrator", orchestrator);
        }

        String taskId = generateDeploymentTask();
        if (testMode) {
            getLog().info(" ... Test mode discovered => displaying and cancelling the generated task");
            getDeploymentHelper().logTaskState(taskId);
            communicator.getProxies().getTaskService().cancel(taskId);
            return;
        }
        runDeploymentTask(taskId);

        if (this.deletePreviouslyDeployedDar && update && currentVersion != null) {
            getLog().info("removing the previous version "+currentVersion);
            try {
                communicator.getProxies().getRepositoryService().delete(currentVersion);
            } catch (Exception e) {
                getLog().error("Cannot delete " + currentVersion + ": " + e.getMessage());
            }
        }
    }

    private ConfigurationItem uploadPackage() {
        final ConfigurationItem deploymentPackage = getDeploymentHelper().uploadPackage(getPackager().perform());
        getLog().info(format("Application ['%s'] has been imported", deploymentPackage.getId()));
        return deploymentPackage;
    }

    private ConfigurationItem getTargetEnvironment() throws MojoExecutionException {
        if (Strings.emptyToNull(environmentId) == null && (environment == null || environment.isEmpty()))
            throw new MojoExecutionException("Mandatory parameter environmentId is not set");

        ConfigurationItem targetEnvironment = getDeploymentHelper().readCiOrNull(environmentId);
        if (targetEnvironment == null) {
            targetEnvironment = createTargetEnvironment();
        }
        getDeploymentHelper().logEnvironment(targetEnvironment);
        return targetEnvironment;
    }

    private void runDeploymentTask(String taskId) throws MojoExecutionException {
        if (skipMode) {
            getLog().info(" ... Skip mode discovered");
            getDeploymentHelper().skipAllSteps(taskId);
        }

        getLog().info("Executing generatedDeployment task");
        try {
            TaskExecutionState taskExecutionState = getDeploymentHelper().executeAndArchiveTask(taskId);
            if (taskExecutionState.isExecutionHalted()) {
                throw new MojoExecutionException(format("Errors when executing task %s. Read logs above to find exact reason.", taskId));
            }
        } catch (IllegalStateException e) {
            if (cancelTaskOnError) {
                getLog().info("cancel task on error " + taskId);
                communicator.getProxies().getTaskService().cancel(taskId);
            }
            throw e;
        }
    }

    private String generateDeploymentTask() throws MojoExecutionException {
        String taskId;
        getLog().info("Creating a task");
        try {
            taskId = communicator.getProxies().getDeploymentService()
                    .createTask(getDeploymentHelper().validateDeployment(generatedDeployment));
        } catch (DeploymentHelper.DeploymentValidationError validationError) {
            for (ValidationMessage validationMessage : validationError.getValidationMessages()) {
                getLog().error(validationMessage.toString());
            }
            throw new RuntimeException(validationError);
        }
        getLog().info("    -> task id: " + taskId);
        return taskId;
    }

    private void generateDeployment(ConfigurationItem deploymentPackage, ConfigurationItem targetEnvironment, Boolean update) {
        DeploymentService deploymentService = communicator.getProxies().getDeploymentService();
        RepositoryService repositoryService = communicator.getProxies().getRepositoryService();

        if (update) {
            getLog().info(" ... Application already exists => preparing update");
            generatedDeployment = deploymentService.prepareUpdate(deploymentPackage.getId(), getDeployedApplicationId(deploymentPackage.getId(), targetEnvironment.getId()));
            currentVersion = repositoryService.read(getDeployedApplicationId(deploymentPackage.getId(), targetEnvironment.getId())).getProperty("version");
            if (generateDeployedOnUpgrade && !hasDeployeds()) {
                generatedDeployment = deploymentService.prepareAutoDeployeds(generatedDeployment);
            }
        } else {
            getLog().info(" ... Application not found in deployed => preparing for initial deployment");
            generatedDeployment = deploymentService.prepareInitial(deploymentPackage.getId(), targetEnvironment.getId());
            generatedDeployment = deploymentService.prepareAutoDeployeds(generatedDeployment);
        }

        if (this.deployeds != null && this.deployeds.size() > 0) {
            getLog().debug("Updating generated deployeds with the configured deployeds");

            generatedDeployment = new CustomDeployedsApplicationStrategy().merge(
                    generatedDeployment, this.deployeds
            );
        }

        if(this.explicitDeployeds) {
            generatedDeployment = getDeploymentHelper().attachExplicitDeployeds(generatedDeployment, this.deployeds);
        }
    }

    private ConfigurationItem createTargetEnvironment() throws MojoExecutionException {
        ConfigurationItem targetEnvironment;
        if (super.environment == null || super.environment.isEmpty()) {
            throw new MojoExecutionException("Environment does not exist and there are no members specified to create it.");
        }
        List<ConfigurationItem> members = Lists.transform(super.environment, new Function<MavenContainer, ConfigurationItem>() {
            @Override
            public ConfigurationItem apply(MavenContainer input) {
                return input.asConfigurationItem();
            }
        });

        try {
            targetEnvironment = getDeploymentHelper().createEnvironment(environmentId, members);
        } catch (DeploymentHelper.EnvironmentAlreadyExistsError e) {
            throw new RuntimeException(e);
        }
        return targetEnvironment;
    }

    private String getDeployedApplicationId(String source, String target) {
        //source = 'Applications/tomcatApps/deployit-petclinic-tomcat/1.0-20120522-173607'
        //target = "Environments/DefaultEnvironment"
        // return "Environments/DefaultEnvironment/deployit-petclinic-tomcat"
        List<String> splitSource =  Lists.newArrayList(Splitter.on("/").split(source));
        final String appName = splitSource.get(splitSource.size() - 2);
        return Joiner.on("/").join(target, appName);
    }

}
