package com.xebialabs.deployit.maven;

import com.google.common.base.Strings;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;

public class MavenDeployable extends Deployable {


    protected String name;
    protected String type;
    protected File location;

    @Deprecated
    protected String entryName;

    protected final Map<String, Object> values = newHashMap();

    protected List<String> tags = newArrayList();

    private String groupId;

    private String artifactId;

    private String classifier;

    private String artifactType;

    private MavenProject project;

    private ArtifactRepository artifactRepository;

    public MavenDeployable() {
    }

    public MavenDeployable(String name, String type) {
        this(name, type, null);
    }

    public MavenDeployable(String name, String type, File location) {
        this.name = name;
        this.type = type;
        this.location = location;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public Map<String, Object> getValues() {
        return values;
    }


    public void setLocation(File location) {
        this.location = location;
    }

    public List<String> getTags() {
        return tags;
    }

    public String getArtifactType() {
        return artifactType;
    }

    public void setArtifactType(String artifactType) {
        this.artifactType = artifactType;
    }

    public String getClassifier() {
        return classifier;
    }

    public void setClassifier(String classifier) {
        this.classifier = classifier;
    }

    public void addTag(String tag) {
        tags.add(tag);
    }


    public void addValue(String name, Object value) {
        checkNotNull(name);
        checkNotNull(value, format("null value for %s", name));
        try {
            if ("location".equals(name)) {
                setLocation(new File((String) value));
            } else {
                Field declaredField = getField(name);
                declaredField.set(this, value);
            }
        } catch (NoSuchFieldException e) {
            values.put(name, value);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(format("Cannot set %s value on field %s", name, value), e);
        }
    }

    private Field getField(String name) throws NoSuchFieldException {
        try {
            return MavenDeployable.class.getDeclaredField(name);
        } catch (NoSuchFieldException nsfe) {
            return MavenDeployable.class.getDeclaredField(name);
        }
    }

    public String getName() {
        return Strings.isNullOrEmpty(name) ? artifactId : name;
    }

    public File getLocation() {
        if (location != null) {
            if (location.isAbsolute()) {
                return location;
            } else {
                return new File(project.getBasedir(), location.getPath());
            }
        } else if (isMavenArtifact()) {
            Artifact artifact = getArtifactRepository().findSingleArtifact(isNullOrEmpty(groupId) ? project.getGroupId() : groupId, artifactId, artifactType, classifier);
            final File file = artifact.getFile();
            if (file == null) {
                throw new IllegalStateException(format("Associated file of %s is empty", artifact));
            }
            return file;
        }
        return null;
    }

    private boolean isMavenArtifact() {
        return !isNullOrEmpty(groupId) || !isNullOrEmpty(artifactId);
    }

    public void setGroupId(String groupId) {
        this.groupId = groupId;
    }

    public void setArtifactId(String artifactId) {
        this.artifactId = artifactId;
    }

    public void setProject(MavenProject project) {
        this.project = project;
    }

    private ArtifactRepository getArtifactRepository() {
        if (project != null && artifactRepository == null) {
            this.artifactRepository = new ArtifactRepository(project);
        }
        return this.artifactRepository;
    }

    @Override
    public String toString() {
        return "MavenDeployable{" +
            "groupId='" + groupId + '\'' +
            ", artifactId='" + artifactId + '\'' +
            ", classifier='" + classifier + '\'' +
            ", artifactType='" + artifactType + '\'' +
            "} " + super.toString();
    }

    public String getArtifactId() {
        return artifactId;
    }

    public static class CiRefProperty {
        private String ref;

        public CiRefProperty(String ref) {
            this.ref = ref;
        }

        public String getRef() {
            return ref;
        }
    }
}
