package com.xebialabs.deployit.maven.packager;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.maven.MavenDeployable;
import com.xebialabs.deployit.maven.converter.Files;

import static java.lang.String.format;

public class MavenDarPackager  {

    private final Log log;

    private final Manifest manifest = new Manifest();

    private final File workingDirectory;

    private final File darFile;


    public MavenDarPackager(MavenProject project, Log log, boolean timestampedVersion) {
        File targetDir = new File(project.getBuild().getDirectory());
        this.log = log;
        this.workingDirectory = new File(targetDir, "deployit-working-dir");
        this.darFile = new File(targetDir, format("%s-%s.dar", project.getArtifactId(), project.getVersion()));
        final Attributes mainAttributes = manifest.getMainAttributes();
        mainAttributes.putValue("Manifest-Version", "1.0");
        mainAttributes.putValue("Deployit-Package-Format-Version", "1.3");
        mainAttributes.putValue("CI-Application", project.getArtifactId());
        mainAttributes.putValue("CI-Version", getCIVersion(project, timestampedVersion));
    }



    public void addDeployable(MavenDeployable deployable) {
        log.debug("Add deployable " + deployable);
        final Map<String, Attributes> entries = manifest.getEntries();
        final Attributes attributes = new Attributes();

        attributes.putValue("CI-Type", deployable.getType());
        attributes.putValue("CI-Name", deployable.getName());
        for (Map.Entry<String, String> me : deployable.getValues().entrySet()) {
            attributes.putValue(me.getKey(), me.getValue());
        }
        entries.put(deployable.getEntryName(), attributes);

        if (!deployable.getTags().isEmpty()) {
            int idx = 0;
            for (String tag : deployable.getTags()) {
                attributes.putValue("CI-tags-EntryValue-" + idx, tag);
                idx++;
            }
        }

        if (deployable.getLocation() != null) {
            final File location = deployable.getLocation();
            if (location.isFile()) {
                log.debug(format(" copy file %s to %s", location, workingDirectory));
                Files.copyFileTo(location, workingDirectory);

            } else {
                log.debug(format(" copy dir  %s to %s", location, workingDirectory));
                Files.copyDirectoryTo(location, workingDirectory);
            }

            if (!location.getName().equals(deployable.getEntryName())) {
                File oldName = new File(workingDirectory, location.getName());
                File newName = new File(workingDirectory, deployable.getEntryName());
                log.debug(format(" rename %s to %s", oldName, newName));
                Files.rename(oldName, newName);
            }
        }
    }

    public void addDeployables(MavenDeployable... deployables) {
        for (MavenDeployable deployable : deployables) {
            addDeployable(deployable);
        }
    }

    public void addDeployables(Collection<? extends MavenDeployable> deployables) {
        for (MavenDeployable deployable : deployables) {
            addDeployable(deployable);
        }
    }

    public File perform() {
        if (!darFile.exists()) {
            log.debug(" create the jar "+darFile);
            Files.jar(darFile, workingDirectory, generateManifestFile());
        }
        return darFile;
    }

    private File generateManifestFile() {
        final File meta_inf = new File(workingDirectory, "META-INF");
        meta_inf.mkdirs();
        File manifestFile = new File(meta_inf, "MANIFEST.MF");
        log.info("Generate manifest file " + manifestFile.getAbsolutePath());
        FileOutputStream fos = null;
        try {
            dumpManifest();
            fos = new FileOutputStream(manifestFile);
            manifest.write(fos);
        } catch (IOException e) {
            throw new RuntimeException("generation of the manifest file failed", e);
        } finally {
            Closeables.closeQuietly(fos);
        }
        return manifestFile;
    }

    private void dumpManifest() throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            manifest.write(baos);
        } finally {
            Closeables.closeQuietly(baos);
        }
        log.debug(new String(baos.toByteArray()));
    }

    private static String getCIVersion(MavenProject project, boolean timestampedVersion) {
        String ciVersion = project.getVersion();
        if (timestampedVersion) {
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd-HHmmss");
            ciVersion = project.getVersion() + "-" + dateFormat.format(System.currentTimeMillis());
            //SNAPSHOT is removed from the timestamped version (as maven deploy does).
            ciVersion = ciVersion.replace("-SNAPSHOT", "");
        }
        return ciVersion;
    }
}
