/*
 * This file is part of Maven Deployit plugin.
 *
 * Maven Deployit plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Maven Deployit plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maven Deployit plugin.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xebialabs.deployit.maven;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.xebialabs.deployit.client.ConnectionOptions;
import com.xebialabs.deployit.client.DarPackager;
import com.xebialabs.deployit.client.DeployitCli;
import com.xebialabs.deployit.client.DeploymentOptions;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.maven.listener.MavenDeploymentListener;
import com.xebialabs.deployit.maven.packager.MavenDarPackager;
import org.apache.maven.artifact.manager.WagonManager;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.wagon.authentication.AuthenticationInfo;

import java.io.File;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static java.lang.String.format;

/**
 * Provides common code for deployit mojos
 *
 * @author Benoit Moussaud
 */
public abstract class AbstractDeployitMojo extends AbstractMojo {


	/**
	 * The maven project.
	 *
	 * @parameter expression="${project}"
	 * @required @readonly
	 */
	protected MavenProject project;


	/**
	 * @parameter default-value="${project.artifactId}"
	 * @required
	 */
	protected String mainArtifactName;


	/**
	 * Activate the skip mode: generate the plan, skip all the steps, validate the task
	 *
	 * @parameter default-value=false
	 */
	protected boolean skipMode;

	/**
	 * Set the orchestrator used during the deployment.
	 *
	 * @parameter default-value=""
	 * @since 3.7.0
	 */
	protected String orchestrator;

	/**
	 * Activate the test mode, generate the plan, display all the steps, validate the task
	 *
	 * @parameter default-value=false
	 */
	protected boolean testMode;

	/**
	 * If a deployments leads no steps, fail the build.
	 *
	 * @parameter default-value=false
	 */
	protected boolean failIfNoStepsAreGenerated;

	/**
	 * Deployit server address
	 *
	 * @parameter default-value="localhost" expression="${deployit.server}"
	 */
	private String serverAddress;

	/**
	 * Deployit Listen port
	 *
	 * @parameter default-value="4516" expression="${deployit.port}"
	 */
	private int port;


	/**
	 * username used to connect to a remote server
	 *
	 * @parameter default-value="" expression="${deployit.username}"
	 */
	private String username;


	/**
	 * password used to connect to a remote server
	 *
	 * @parameter default-value="" expression="${deployit.password}"
	 */
	private String password;

	/**
	 * context of the deployit server
	 *
	 * @parameter default-value="deployit" expression="${deployit.context}"
	 */
	private String context;

	/**
	 * id of the server in the setting.xml file
	 *
	 * @parameter default-value="deployit-credentials" expression="${deployit.credentials}"
	 * @since 3.6.4
	 */
	private String server;

	/**
	 * If true the communication with the deployit server is secured (https).
	 *
	 * @parameter default-value=false
	 */
	protected boolean secured;


	/**
	 * Id of the environment used for the deployment.
	 *
	 * @parameter default-value="" expression="${deployit.environmentId}"
	 */
	private String environmentId;

	/**
	 * List of the deployeds: extensions or complete if you use explicitDeployeds options
	 *
	 * @parameter
	 */
	protected List<MavenDeployed> deployeds;

	/**
	 * List of the deployables, including artifacts or middleware resource specification (eg Datasource)
	 *
	 * @parameter
	 */
	protected List<MavenDeployable> deployables = newArrayList();

	/**
	 * List of container in the target environment, if you want to create the environment through the Maven plugin.
	 *
	 * @parameter
	 */
	protected List<MavenContainer> environment;

	/**
	 * Use this attribute to add a timestamp to the version of the deployit package.
	 * by default,the SNAPSHOT versions are automatically timestamped. This flag is useful only if you want to timestamp non SNAPSHOT version.
	 *
	 * @parameter default-value = false  expression="${deployit.timestamp}"
	 */
	protected boolean timestampedVersion;

	/**
	 * Delete the previous deployed dar. Useful if you work with the SNAPSHOT versions you don't want to keep in your repository.
	 *
	 * @parameter default-value=false  expression="${deployit.delete.previous.dar}"
	 */
	protected boolean deletePreviouslyDeployedDar;

	/**
	 * Flag controlling whether, during the upgrade operation, the Deployed objects should generated (like an initial deployment) or reused.
	 * For security reasons, the default value is false but should be set to true to apply the modifications (new Ear, removed links) even during upgrade.
	 *
	 * @parameter default-value=false  expression="${deployit.generate.deployed.on.upgrade}"
	 * @since 3.6.2
	 */
	protected boolean generateDeployedOnUpgrade;

	/**
	 * When a task falls in error, it is cancelled. Sometime it could be interesting to debug it using the UI or the CLI. When this flag is set to false, the
	 * task will be left as is.
	 *
	 * @parameter default-value=true  expression="${deployit.cancel.task.on.error}"
	 * @since 3.6.2
	 */
	protected boolean cancelTaskOnError;

	/**
	 * With explicitDeployeds true, the deployed are not generated but fully loaded from the pluging configuration.
	 *
	 * @parameter default-value=false
	 */
	protected boolean explicitDeployeds;

	private DarPackager packager;

	protected DeployitCli deployitClient;

	/**
	 * The Maven Wagon manager to use when obtaining server authentication details.
	 *
	 * @component role="org.apache.maven.artifact.manager.WagonManager"
	 * @required
	 * @readonly
	 */
	private WagonManager wagonManager;

	protected DeployitCli getClient() {
		if (deployitClient == null) {
			ConnectionOptions options = new ConnectionOptions();
			options.setHost(serverAddress);
			options.setPort(port);
			options.setUsername(getUsername());
			options.setPassword(getPassword());
			options.setContext(context);
			options.setSecured(secured);
			deployitClient = new DeployitCli(options, new MavenDeploymentListener(getLog()));
		}
		return deployitClient;
	}

	protected void deploy() throws MojoExecutionException {
		final RepositoryObject deploymentPackage = importDar();
		final RepositoryObject environment = fetchEnvironment();

		DeploymentOptions deploymentOptions = new DeploymentOptions();
		deploymentOptions.setSkipMode(skipMode);
		deploymentOptions.setTestMode(testMode);
		deploymentOptions.setFailIfNoStepsAreGenerated(failIfNoStepsAreGenerated);
		deploymentOptions.setDeletePreviouslyDeployedArtifact(deletePreviouslyDeployedDar);
		deploymentOptions.setExplicitMappings(explicitDeployeds);
		deploymentOptions.setGenerateDeployedOnUpgrade(generateDeployedOnUpgrade);
		deploymentOptions.setCancelTaskOnError(cancelTaskOnError);
		deploymentOptions.setOrchestrator(orchestrator);

		getLog().info(format("-- Deploy %s on %s", deploymentPackage.getId(), environment.getId()));
		getClient().deploy(deploymentPackage.getId(),
				environment.getId(), deployeds, deploymentOptions, new MavenDeploymentListener(getLog()));
	}

	protected RepositoryObject importDar() throws MojoExecutionException {
		final File darFile = getPackager().perform();
		getLog().info("Import dar file " + darFile);
		return getClient().importPackage(darFile.getPath(), new MavenDeploymentListener(getLog()));
	}

	protected void undeploy() throws MojoExecutionException {
		getClient().undeployAndWait(environmentId + "/" + mainArtifactName);
	}

	protected RepositoryObject fetchEnvironment() throws MojoExecutionException {

		if (Strings.emptyToNull(environmentId) == null)
			throw new MojoExecutionException("environmentId is not set");

		try {
			getLog().info("read the environment " + environmentId);
			final RepositoryObject repositoryObject = getClient().get(environmentId);
			if (getLog().isDebugEnabled() && repositoryObject != null) {
				getLog().debug(" dump members of " + environmentId);
				for (Object m : (List) repositoryObject.getValues().get("members"))
					getLog().debug("  -- member " + m);
			}
			return repositoryObject;
		} catch (Exception e) {
			getLog().debug(e.getMessage());
			if (environment == null)
				throw new MojoExecutionException("Cannot fetch environment " + environmentId + " and no members are defined in <environnment>", e);

			getLog().info("Create the members of environment");
			List<String> members = newArrayList();
			for (MavenContainer each : environment) {
				getLog().info(" create " + each.getId());
				getClient().create(each);
				if (each.isAddedToEnvironment())
					members.add(each.getId());
			}

			getLog().info("Create environment " + environmentId);
			MavenContainer ciEnvironment = new MavenContainer();
			ciEnvironment.setId(environmentId);
			ciEnvironment.setType("Environment");
			ciEnvironment.addParameter("members", members);
			return getClient().create(ciEnvironment);
		}
	}


	DarPackager getPackager() {
		if (packager == null) {
			packager = new MavenDarPackager(project, getLog(), timestampedVersion || isSnapshotVersion());
			if (deployables.isEmpty())
				addMainArtifact();
			//Inject the project into the MavenDeployables
			packager.addDeployables(transform(deployables, new Function<MavenDeployable, MavenDeployable>() {
				@Override
				public MavenDeployable apply(MavenDeployable input) {
					input.setProject(project);
					return input;
				}
			}));
		}
		return packager;
	}

	private void addMainArtifact() {
		final MavenDeployable mainArtifactDeployable;
		try {
			mainArtifactDeployable = generateMainDeployable();
			if (mainArtifactDeployable != null) {
				deployables.add(mainArtifactDeployable);
			}
		} catch (MojoExecutionException e) {
			e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
		}
	}

	private MavenDeployable generateMainDeployable() throws MojoExecutionException {

		final String packaging = project.getPackaging();
		if (packaging.equalsIgnoreCase("dar") || packaging.equalsIgnoreCase("pom")) {
			getLog().debug("No main artifact for this packaging " + packaging);
			return null;
		}
		String ciType;
		if (packaging.equalsIgnoreCase("war")) {
			ciType = "jee.War";
		} else if (packaging.equalsIgnoreCase("ear")) {
			ciType = "jee.Ear";
		} else {
			getLog().error("Unsupported pom packaging type. You must supply the deployable definition for the main artifact.");
			return null;
		}

		MavenDeployable mainArtifact = new MavenDeployable();
		mainArtifact.setType(ciType);
		mainArtifact.setGroupId(project.getGroupId());
		mainArtifact.setArtifactId(mainArtifactName);
		final File file = project.getArtifact().getFile();
		if (file != null) {
			mainArtifact.setLocation(file);
		} else {
			throw new MojoExecutionException("Cannot automatically discovery main artifact file location. You must supply the deployable definition for the main artifact.");
		}
		return mainArtifact;
	}

	private boolean isSnapshotVersion() {
		return project.getVersion().contains("SNAPSHOT");
	}

	public String getUsername() {
		if (this.username == null) {
			getLog().debug("No username defined in pom.xml and " +
					"no system property defined either. Trying to look up " +
					"username in settings.xml under server element " + server);
			AuthenticationInfo authenticationInfo = this.wagonManager.getAuthenticationInfo(server);
			if (authenticationInfo == null) {
				getLog().warn(format("In settings.xml server element '%s' was not defined. Using default username.", server));
				return null;
			}
			if (authenticationInfo.getUserName() != null) {
				getLog().debug(format(" username '%s' found in the settings.xml file", authenticationInfo.getUserName()));
				return authenticationInfo.getUserName();
			} else {
				getLog().warn(format(
						"In settings.xml no username was found for server element '%s'. Does the element exist? Using default username.", server));
				return null;
			}
		}
		return username;
	}

	public String getPassword() {
		if (this.password == null) {
			getLog().debug("No password defined in pom.xml and " +
					"no system property defined either. Trying to look up " +
					"password in settings.xml under server element " + server);
			AuthenticationInfo authenticationInfo = this.wagonManager.getAuthenticationInfo(server);
			if (authenticationInfo == null) {
				getLog().warn(format("In settings.xml server element '%s' was not defined. Using default password.", server));
				return null;
			}
			if (authenticationInfo.getPassword() != null) {
				getLog().debug(format(" password '%s' found in the settings.xml file", "********"));
				return authenticationInfo.getPassword();
			} else {
				getLog().warn(format(
						"In settings.xml no password was found for server element '%s'. Does the element exist? Using default password", server));
				return null;
			}
		}
		return password;
	}

}
