package com.xebialabs.deployit.maven;

import com.google.common.base.Strings;
import com.xebialabs.deployit.client.Deployable;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.lang.reflect.Field;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;
import static java.lang.String.format;

public class MavenDeployable extends Deployable {

	private String groupId;

    private String artifactId;

	private MavenProject project;

	public void addValue(String name, String value) {
		checkNotNull(name);
		checkNotNull(value, format("null value for %s", name));
		try {
			if ("location".equals(name)) {
				setLocation(new File(value));
			} else {
				Field declaredField = getField(name);
				declaredField.set(this, value);
			}
		} catch (NoSuchFieldException e) {
			values.put("CI-" + name, value);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(format("Cannot set %s value on field %s", name, value), e);
		}
	}

	private Field getField(String name) throws NoSuchFieldException {
		try {
			return MavenDeployable.class.getDeclaredField(name);
		} catch (NoSuchFieldException nsfe) {
			return Deployable.class.getDeclaredField(name);
		}
	}

	@Override
	public String getName() {
		return Strings.isNullOrEmpty(name) ? artifactId : name;
	}

	@Override
	public File getLocation() {
		if (location != null) {
			if (location.isAbsolute())
				return location;
			else
				return new File(project.getBasedir(), location.getPath());
		} else if (isMavenArtefact()) {
			String key = groupId + ":" + artifactId;
			Artifact artifact = (Artifact) project.getArtifactMap().get(key);
			final Set dependencyArtifacts = project.getDependencyArtifacts();
			if (dependencyArtifacts != null) {
				for (Object o : dependencyArtifacts) {
					Artifact da = (Artifact) o;
					final String artifactKey = da.getGroupId() + ":" + da.getArtifactId();
					if (artifactKey.equals(key)) {
						artifact = da;
					}
				}
			}
			if (artifact == null)
				throw new IllegalStateException(format("The artifact %s not found in the project dependencies %s", key, dependencyArtifacts));


			final File file = artifact.getFile();
			if (file == null) {
				throw new IllegalStateException(format("Associated file of %s is empty", artifact));
			}

			return file;
		}
		return null;
	}

	private boolean isMavenArtefact() {
		return !isNullOrEmpty(groupId) && !isNullOrEmpty(artifactId);
	}

	public void setGroupId(String groupId) {
		this.groupId = groupId;
	}

	public void setArtifactId(String artifactId) {
		this.artifactId = artifactId;
	}

	public void setProject(MavenProject project) {
		this.project = project;
	}

	@Override
	public String toString() {
		return "MavenDeployable{" +
				"artifactId='" + artifactId + '\'' +
				", groupId='" + groupId + '\'' +
				", project=" + project +
				"} " + super.toString();
	}

	public String getArtifactId() {
		return artifactId;
	}
}
