/*
 * This file is part of Maven Deployit plugin.
 *
 * Maven Deployit plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Maven Deployit plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maven Deployit plugin.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xebialabs.deployit.maven.converter;

import com.xebialabs.deployit.maven.Container;
import com.xebialabs.deployit.maven.Deployable;
import com.xebialabs.deployit.maven.Deployed;
import org.codehaus.plexus.component.configurator.ComponentConfigurationException;
import org.codehaus.plexus.component.configurator.ConfigurationListener;
import org.codehaus.plexus.component.configurator.converters.AbstractConfigurationConverter;
import org.codehaus.plexus.component.configurator.converters.ConfigurationConverter;
import org.codehaus.plexus.component.configurator.converters.lookup.ConverterLookup;
import org.codehaus.plexus.component.configurator.expression.ExpressionEvaluator;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.configuration.PlexusConfigurationException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * User: bmoussaud
 * Date: 21 avr. 2010
 * Time: 13:28:04
 */
public class DeployitCIConverter extends AbstractConfigurationConverter {

    public static final String MAVEN_EXPRESSION_EVALUATOR_ID = "maven.expressionEvaluator";

    public static final String ROLE = ConfigurationConverter.class.getName();

    static final String ADD_TO_ENV = "addToEnv";

    public boolean canConvert(Class type) {
        return Container.class.isAssignableFrom(type) || Deployed.class.isAssignableFrom(type) || Deployable.class.isAssignableFrom(type);
    }

    /**
     * @see org.codehaus.plexus.component.configurator.converters.ConfigurationConverter#fromConfiguration(org.codehaus.plexus.component.configurator.converters.lookup.ConverterLookup, org.codehaus.plexus.configuration.PlexusConfiguration, java.lang.Class, java.lang.Class, java.lang.ClassLoader, org.codehaus.plexus.component.configurator.expression.ExpressionEvaluator, org.codehaus.plexus.component.configurator.ConfigurationListener)
     */
    public Object fromConfiguration(ConverterLookup converterLookup, PlexusConfiguration configuration, Class type,
                                    Class baseType, ClassLoader classLoader, ExpressionEvaluator expressionEvaluator,
                                    ConfigurationListener listener)
            throws ComponentConfigurationException {

        if (type.equals(Container.class)) {
            return convertToContainer(configuration);
        }

        if (type.equals(Deployable.class)) {
            return convertToDeployable(configuration);
        }

        if (type.equals(Deployed.class)) {
            return convertToDeployed(configuration);
        }

        throw new ComponentConfigurationException("type not handled (" + type + ")");


    }

    private Container convertToContainer(PlexusConfiguration configuration) throws ComponentConfigurationException {
        Container container = new Container();
        try {
            Map<String, Object> properties = doConvertToProperties(configuration);
            for (Map.Entry<String, Object> entry : properties.entrySet()) {
                container.addParameter(entry.getKey(), entry.getValue());
            }
            final String add2Env = configuration.getAttribute(ADD_TO_ENV);
            if (add2Env != null)
                container.setAddedToEnvironment(Boolean.parseBoolean(add2Env));
        } catch (Exception e) {
            throw new ComponentConfigurationException(e);
        }
        return container;
    }

    private Map<String, Object> doConvertToProperties(PlexusConfiguration configuration) throws PlexusConfigurationException, ComponentConfigurationException {
        Map<String, Object> properties = new HashMap<String, Object>();
        for (PlexusConfiguration propertyConfiguration : configuration.getChildren()) {
            String name = propertyConfiguration.getName();
            if (propertyConfiguration.getChild("list", false) != null) {
                PlexusConfiguration[] values = propertyConfiguration.getChild("list").getChildren();
                List<String> list = new ArrayList<String>();
                for (int i = 0; i < values.length; i++) {
                    PlexusConfiguration valueConfiguration = values[i];
                    list.add(valueConfiguration.getValue());
                }
                properties.put(name, list);
            } else if (propertyConfiguration.getChild("map", false) != null) {
                PlexusConfiguration[] entries = propertyConfiguration.getChild("map").getChildren();
                Map<String, String> map = new HashMap<String, String>();
                for (int i = 0; i < entries.length; i++) {
                    PlexusConfiguration entryConfiguration = entries[i];
                    map.put(entryConfiguration.getName(), entryConfiguration.getValue());
                }
                properties.put(name, map);
            } else if (name.equals("placeholders")) {
                properties.put(name, convertOldPlaceholderFormat(propertyConfiguration));
            } else {
                properties.put(name, propertyConfiguration.getValue());
            }
        }
        return properties;
    }

    private Map<String,String> convertOldPlaceholderFormat(PlexusConfiguration configuration) throws ComponentConfigurationException {
        Map<String, String> map = new HashMap<String, String>();
        for (PlexusConfiguration entry : configuration.getChildren()) {
            final PlexusConfiguration k = entry.getChild("k");
            final PlexusConfiguration v = entry.getChild("v");
            try {
                map.put(k.getValue(), v.getValue());
            } catch (PlexusConfigurationException e) {
                throw new ComponentConfigurationException("kv getValue error", e);
            }
        }
        return map;
    }

    private Deployed convertToDeployed(PlexusConfiguration configuration) throws ComponentConfigurationException {
        try {
            Deployed deployed = new Deployed();
            Map<String, Object> properties = doConvertToProperties(configuration);
            for (Map.Entry<String, Object> entry : properties.entrySet()) {
                if (entry.getKey().equals("placeholders")) {
                    deployed.getPlaceholders().putAll((Map<String, String>) entry.getValue());
                } else {
                    deployed.addValues(entry.getKey(), entry.getValue());
                }
            }
            return deployed;
        } catch (Exception e) {
            throw new ComponentConfigurationException(e);
        }
    }

    private Deployable convertToDeployable(PlexusConfiguration configuration) throws ComponentConfigurationException {
        try {
            return doConvertToDeployable(configuration);
        } catch (Exception e) {
            throw new ComponentConfigurationException(e);
        }
    }

    private Deployable doConvertToDeployable(PlexusConfiguration configuration) throws PlexusConfigurationException {
        Deployable deployable = new Deployable();
        for (PlexusConfiguration propertyConfiguration : configuration.getChildren()) {
            String name = propertyConfiguration.getName();
            if (propertyConfiguration.getChild("list", false) != null) {
                PlexusConfiguration[] values = propertyConfiguration.getChild("list").getChildren();
                for (int i = 0; i < values.length; i++) {
                    PlexusConfiguration valueConfiguration = values[i];
                    String indexName = name + "-EntryValue-" + i;
                    deployable.addValue(indexName, valueConfiguration.getValue());
                }
            } else if (propertyConfiguration.getChild("map", false) != null) {
                PlexusConfiguration[] entries = propertyConfiguration.getChild("map").getChildren();
                for (int i = 0; i < entries.length; i++) {
                    PlexusConfiguration entryConfiguration = entries[i];
                    String indexName = name + "-" + entryConfiguration.getName();
                    deployable.addValue(indexName, entryConfiguration.getValue());
                }
            } else {
                String c = propertyConfiguration.getValue();
                deployable.addValue(name, c);
            }
        }
        return deployable;
    }

}

