package com.xebialabs.deployit.maven;

import com.google.common.collect.Maps;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.lang.reflect.Field;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.isNullOrEmpty;
import static java.lang.String.format;

public class Deployable {

	private String type;

	private String location;

	private String name;

	private String groupId;

	private String artifactId;

	private String fileSystemLocation;

	private String entryName;

	final private Map<String, String> values = Maps.newHashMap();

	public void addValue(String name, String value) {
		checkNotNull(name);
		checkNotNull(value, format("null value for %s", name));
		try {
			Field declaredField = Deployable.class.getDeclaredField(name);
			declaredField.set(this, value);
		} catch (NoSuchFieldException e) {
			values.put("CI-" + name, value);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(format("Cannot set %s value on field %s", name, value), e);
		}
	}

	@Override
	public String toString() {
		return "Deployable{" +
				"type='" + type + '\'' +
				", location='" + location + '\'' +
				", name='" + name + '\'' +
				", groupId='" + groupId + '\'' +
				", artifactId='" + artifactId + '\'' +
				", fileSystemLocation='" + fileSystemLocation + '\'' +
				", entryName='" + entryName + '\'' +
				", values=" + values +
				'}';
	}

	public void consolidate(MavenProject project) {
		if (isMavenArtefact()) {
			if (isNullOrEmpty(location)) {
				String key = groupId + ":" + artifactId;
				Artifact artifact = (Artifact) project.getArtifactMap().get(key);
				final Set dependencyArtifacts = project.getDependencyArtifacts();
				if (dependencyArtifacts != null) {
					for (Object o : dependencyArtifacts) {
						Artifact da = (Artifact) o;
						final String artifactKey = da.getGroupId() + ":" + da.getArtifactId();
						if (artifactKey.equals(key)) {
							artifact = da;
						}
					}
				}
				if (artifact == null)
					throw new IllegalStateException(format("The artifact %s  not found in the project dependencies %s", key, dependencyArtifacts));


				final File file = artifact.getFile();
				if (file == null) {
					throw new IllegalStateException(format("Associated file of %s is empty", artifact));
				}

				final String artifactFile = file.toString();
				location = artifactFile;
				fileSystemLocation = artifactFile;
			}
			if (isNullOrEmpty(name)) {
				name = artifactId;
			}
			if (isNullOrEmpty(entryName)) {
				entryName = new File(fileSystemLocation).getName();
			}
		} else if (isFileSystemLocation(project)) {
			final File fileLocation = new File(location);
			final File fileSysLoca;
			if (fileLocation.isAbsolute()) {
				fileSysLoca = fileLocation;
			} else {
				fileSysLoca = new File(project.getBasedir(), location);
			}
			fileSystemLocation = fileSysLoca.getPath();
			if (isNullOrEmpty(entryName)) {
				if (fileSysLoca.isDirectory()) {
					entryName = fileSysLoca.getName();
				} else {
					entryName = location;
				}
			}
		} else {
			//Resource
			if (isNullOrEmpty(entryName)) {
				checkNotNull(name, "name should not be null on this %s", this);
				entryName = name;
			}
		}

	}

	private boolean isFileSystemLocation(MavenProject project) {
		if (isNullOrEmpty(location))
			return false;
		return new File(location).exists() || new File(project.getBasedir(), location).exists();
	}

	private boolean isMavenArtefact() {
		return !isNullOrEmpty(groupId) && !isNullOrEmpty(artifactId);
	}

	public String getFileSystemLocation() {
		return fileSystemLocation;
	}

	public Map<String, String> getValues() {
		return values;
	}

	public String getType() {
		return type;

	}

	public String getName() {
		return name;
	}

	public String getEntryName() {
		return entryName;
	}

	public void setType(String type) {
		this.type = type;
	}

	public void setLocation(String location) {
		this.location = location;
	}

	public void setName(String name) {
		this.name = name;
	}

	public void setGroupId(String groupId) {
		this.groupId = groupId;
	}

	public void setArtifactId(String artifactId) {
		this.artifactId = artifactId;
	}

	public void setFileSystemLocation(String fileSystemLocation) {
		this.fileSystemLocation = fileSystemLocation;
	}
}
