/*
 * This file is part of Maven Deployit plugin.
 *
 * Maven Deployit plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Maven Deployit plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Maven Deployit plugin.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xebialabs.deployit.maven;

import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.maven.cli.MavenCli;
import com.xebialabs.deployit.maven.cli.MavenCliOptions;
import com.xebialabs.deployit.maven.packager.ManifestPackager;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Provides common code for deployit mojos
 *
 * @author Benoit Moussaud
 */
public abstract class AbstractDeployitMojo extends AbstractMojo {


	/**
	 * The maven project.
	 *
	 * @parameter expression="${project}"
	 * @required @readonly
	 */
	protected MavenProject project;


	/**
	 * @parameter default-value="${project.artifactId}"
	 * @required
	 */
	protected String mainArtifactName;


	/**
	 * Activate the skip mode: generate the plan, skip all the steps, validate the task
	 *
	 * @parameter default-value=false
	 */
	protected boolean skipMode;

	/**
	 * Activate the test mode, generate the plan, display all the steps, validate the task
	 *
	 * @parameter default-value=false
	 */
	protected boolean testMode;

	/**
	 * If a deployments leads no steps, fail the build.
	 *
	 * @parameter default-value=false
	 */
	protected boolean failIfNoStepsAreGenerated;

	/**
	 * Deployit server address
	 *
	 * @parameter default-value="localhost" expression="${deployit.server}"
	 */
	private String serverAddress;

	/**
	 * Deployit Listen port
	 *
	 * @parameter default-value="4516" expression="${deployit.port}"
	 */
	private int port;


	/**
	 * username used to connect to a remote server
	 *
	 * @parameter default-value="" expression="${deployit.username}"
	 */
	private String username;


	/**
	 * password used to connect to a remote server
	 *
	 * @parameter default-value="" expression="${deployit.password}"
	 */
	private String password;

	/**
	 * context of the deployit server
	 *
	 * @parameter default-value="deployit" expression="${deployit.context}"
	 */
	private String context;

	/**
	 * If true the communication with the deployit server is secured (https).
	 *
	 * @parameter default-value=false
	 */
	protected boolean secured;


	/**
	 * Id of the environment used for the deployment.
	 *
	 * @parameter default-value="" expression="${deployit.environmentId}"
	 */
	private String environmentId;

	/**
	 * List of the deployeds: extensions or complete if you use explicitDeployeds options
	 *
	 * @parameter
	 */
	protected List<Deployed> deployeds;

	/**
	 * List of the deployables, including artifacts or middleware resource specification (eg Datasource)
	 *
	 * @parameter
	 */
	protected List<Deployable> deployables;


	/**
	 * List of container in the target environment, if you want to create the environment through the Maven plugin.
	 *
	 * @parameter
	 */
	protected List<Container> environment;

	/**
	 * Use this attribute to add a timestamp to the version of the deployit package.
	 * by default,the SNAPSHOT versions are automatically timestamped
	 *
	 * @parameter default-value = false  expression="${deployit.timestamp}"
	 */
	protected boolean timestampedVersion;

	/**
	 * Delete the previous deployed dar. Useful if you work with the SNAPSHOT versions you don't want to keep in your repository.
	 *
	 * @parameter default-value=false  expression="${deployit.delete.previous.dar}"
	 */
	protected boolean deletePreviouslyDeployedDar;

	/**
	 * With explicitDeployeds true, the deployed are not generated but fully loaded from the pluging configuration.
	 *
	 * @parameter default-value=false
	 */
	protected boolean explicitDeployeds;

	protected ManifestPackager packager;

	protected MavenCli client;

	protected MavenCli getClient() throws MojoExecutionException {
		if (client == null) {
			MavenCliOptions options = new MavenCliOptions();
			options.setHost(serverAddress);
			options.setPort(port);
			options.setUsername(username);
			options.setPassword(password);
			options.setSkipMode(skipMode);
			options.setTestMode(testMode);
			options.setContext(context);
			options.setExplicitMappings(explicitDeployeds);
			options.setSecured(secured);
			options.setFailIfNoStepsAreGenerated(failIfNoStepsAreGenerated);
			try {
				client = new MavenCli(options, getLog());
			} catch (Exception e) {
				throw new MojoExecutionException("Error when creating MavenCli", e);
			}

		}
		return client;
	}

	protected void deploy() throws MojoExecutionException {
		final RepositoryObject deploymentPackage = importDar();

		final RepositoryObject environment = fetchEnvironment();

		getLog().info(String.format("-- Deploy %s on %s", deploymentPackage.getId(), environment.getId()));
		final String previousPackageId = getClient().deploy(deploymentPackage.getId(), environment.getId(), deployeds);

		if (shouldDeletePreviousVersion(previousPackageId)) {
			getLog().info("Delete previously deployed dar " + previousPackageId);
			getClient().delete(previousPackageId);
		}
	}

	private boolean shouldDeletePreviousVersion(String previousPackageId) {
		if (testMode)
			return false;

		if (skipMode)
			return false;

		return deletePreviouslyDeployedDar && StringUtils.isNotBlank(previousPackageId);
	}

	protected RepositoryObject importDar() throws MojoExecutionException {
		final File darFile = getPackager().getDarFile();
		if (!darFile.exists()) {
			getLog().info("Dar file does not exist " + darFile);
			getLog().info("generate it...");
			getPackager().perform();
			getPackager().seal();
		}
		return importDar(darFile);
	}

	protected void undeploy() throws MojoExecutionException {
		getClient().undeployAndWait(environmentId + "/" + mainArtifactName);
	}

	protected RepositoryObject importDar(File darFile) throws MojoExecutionException {
		getLog().info("Import dar file " + darFile);
		return getClient().importPackage(darFile);
	}

	protected RepositoryObject fetchEnvironment() throws MojoExecutionException {

		if (StringUtils.isBlank(environmentId))
			throw new MojoExecutionException("environmentId is not set");

		try {
			getLog().info("read the environment " + environmentId);
			final RepositoryObject repositoryObject = getClient().get(environmentId);
			if (getLog().isDebugEnabled() && repositoryObject != null) {
				getLog().debug(" dump members of " + environmentId);
				for (Object m : (List) repositoryObject.getValues().get("members"))
					getLog().debug("  -- member " + m);
			}
			return repositoryObject;
		} catch (Exception e) {
			getLog().debug(e.getMessage());
			if (environment == null)
				throw new MojoExecutionException("Cannot fetch environment " + environmentId + " and no members are defined in <environnment>", e);

			getLog().info("Create the members of environment");
			List<String> members = newArrayList();
			for (Container each : environment) {
				getLog().info(" create " + each.getId());
				getClient().create(each);
				if (each.isAddedToEnvironment())
					members.add(each.getId());
			}

			getLog().info("Create environment " + environmentId);
			Container ciEnvironment = new Container();
			ciEnvironment.setId(environmentId);
			ciEnvironment.setType("Environment");
			ciEnvironment.addParameter("members", members);
			return getClient().create(ciEnvironment);
		}
	}

	ManifestPackager getPackager() {
		if (packager == null) {
			packager = new ManifestPackager(project, getLog(), timestampedVersion || isSnapshotVersion());
			if (project.getArtifact().getType().equalsIgnoreCase("war") || project.getArtifact().getType().equalsIgnoreCase("ear")) {
				Deployable mainArtifact = new Deployable();
				mainArtifact.setGroupId(project.getGroupId());
				mainArtifact.setArtifactId(mainArtifactName);
				if (project.getArtifact().getType().equalsIgnoreCase("war")) {
					mainArtifact.setType("jee.War");
				} else {
					mainArtifact.setType("jee.Ear");
				}
				final File file = project.getArtifact().getFile();
				if (file != null) {
					mainArtifact.setFileSystemLocation(file.toString());
					mainArtifact.setLocation(file.toString());
				}

				if (deployables == null) {
					deployables = newArrayList();
				}

				deployables.add(mainArtifact);
			}
			packager.addDeployables(deployables);
		}
		return packager;
	}

	private boolean isSnapshotVersion() {
		return project.getVersion().contains("SNAPSHOT");
	}
}
