package com.xebialabs.license.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Map;
import java.util.UUID;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.HexEncoder;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;

import com.xebialabs.license.*;

import static com.xebialabs.license.LicenseProperty.*;
import static com.xebialabs.license.LicenseReader.*;

public class LicenseGenerator {
    private final static String PRIV_KEY = "3082014b0201003082012c06072a8648ce3804013082011f02818100fd7f53811d75122952df4a9c2eece4e7f611b7523cef4400c31e3f80b6512669455d402251fb593d8d58fabfc5f5ba30f6cb9b556cd7813b801d346ff26660b76b9950a5a49f9fe8047b1022c24fbba9d7feb7c61bf83b57e7c6a8a6150f04fb83f6d3c51ec3023554135a169132f675f3ae2b61d72aeff22203199dd14801c70215009760508f15230bccb292b982a2eb840bf0581cf502818100f7e1a085d69b3ddecbbcab5c36b857b97994afbbfa3aea82f9574c0b3d0782675159578ebad4594fe67107108180b449167123e84c281613b7cf09328cc8a6e13c167a8b547c8d28e0a3ae1e2bb3a675916ea37f0bfa213562f1fb627a01243bcca4f1bea8519089a883dfe15ae59f06928b665e807b552564014c3bfecf492a041602140480cc8eb9735d2d83f0a5f4ffd15aa3c76d7298";

    private static void writeKey(Key key) throws IOException {
        byte[] bytes = key.getEncoded();
        HexEncoder hexit = new HexEncoder();
        hexit.encode(bytes, 0, bytes.length, System.out);
    }

    /**
     * Utility to generate a new key pair. Mostly for testing.
     */
    private static void genKeyPair(int keySize) throws IOException, GeneralSecurityException {
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("DSA", "BC");
        SecureRandom random = SecureRandom.getInstance("SHA1PRNG", "BC");

        keyGen.initialize(keySize, random);
        KeyPair pair = keyGen.generateKeyPair();

        PrivateKey priv = pair.getPrivate();
        System.out.print("Private key: ");
        writeKey(priv);

        PublicKey pub = pair.getPublic();
        System.out.print("Public key : ");
        writeKey(pub);
    }

    private static void writeSigned(License license, String outFile) throws IOException, GeneralSecurityException {
        try (OutputStream signedFile = new FileOutputStream(outFile)) {
            writeSigned(license, signedFile);
        }
    }

    private static void writeSigned(License license, OutputStream signedFile) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeySpecException, IOException, InvalidKeyException, SignatureException {
        String lineSeparator = System.getProperty("line.separator");

        KeyFactory kf = KeyFactory.getInstance("DSA", "BC");
        PrivateKey privateKey = kf.generatePrivate(new PKCS8EncodedKeySpec(readKey(PRIV_KEY)));

        Signature dsa = Signature.getInstance("SHA1withDSA", "BC");
        dsa.initSign(privateKey);

        byte[] data = license.toLicenseContent().getBytes(LICENSE_FILE_ENCODING);
        dsa.update(data);

        byte[] signature = dsa.sign();

        String version = license.getLicenseVersion();
        String preamble = version.equals(LicenseVersion1.VERSION) ?
                LICENSE_PREAMBLE_V1 : LICENSE_PREAMBLE;

        signedFile.write(preamble.getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(data);

        signedFile.write(String.format(SIGNATURE_PREAMBLE, dsa.getAlgorithm()).getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));
        HexEncoder hexit = new HexEncoder();
        hexit.encode(signature, 0, signature.length, new LineBreakingOutputStream(signedFile));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));

        signedFile.write(SIGNATURE_ENDING.getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));
    }

    public static boolean validateLicense(String licenseFilename) throws IOException {
        try {
            License license = readLicense(new File(licenseFilename));
            System.out.printf(license.toString());
            return true;
        } catch (InvalidLicenseException e) {
            System.out.println("License invalid: " + e.getMessage());
        } catch (UnableToValidateLicenseException e) {
            System.out.println("License check failed: " + e.getMessage());
        }
        return false;
    }

    public static void genLicense(String licenseVersion, String file, Multimap<LicenseProperty, String> licenseProperties) throws IOException, GeneralSecurityException {
        License license;
        if (LicenseVersion1.VERSION.equals(licenseVersion)) {
            license = new LicenseVersion1(licenseProperties);
        } else if (Strings.isNullOrEmpty(licenseVersion) || LicenseVersion2.VERSION.equals(licenseVersion)) {
            licenseProperties.put(LICENSE_VERSION, LicenseVersion2.VERSION);
            license = new LicenseVersion2(licenseProperties);
        } else {
            throw new InvalidLicenseException(String.format("Unsupported license version %s", licenseVersion));
        }
        writeSigned(license, file);
    }

    public static void main(String[] args) throws IOException, GeneralSecurityException {
        // ensure we have bouncy castle
        Security.addProvider(new BouncyCastleProvider());

        LicenseGeneratorOptions options = LicenseGeneratorOptions.parseCommandLine(args);
        if (options == null) {
            return;
        }

        switch (options.getMode()) {
            case GENERATE_KEYPAIR:
                genKeyPair(options.getKeySize());
                break;
            case GENERATE_LICENSE:
                Multimap<LicenseProperty, String> licenseProperties = getLicenseProperties(options);
                genLicense(options.getLicenseVersion(), options.getFile(), licenseProperties);
                break;
            case VALIDATE_KEY:
                validateLicense(options.getFile());
                break;
        }
    }

    private static Multimap<LicenseProperty, String> getLicenseProperties(LicenseGeneratorOptions options) {
        Multimap<LicenseProperty, String> licenseProperties = ArrayListMultimap.create();
        licenseProperties.put(PRODUCT, options.getProduct());
        licenseProperties.put(LICENSED_TO, options.getLicensedTo());
        licenseProperties.put(CONTACT, options.getContact());
        licenseProperties.put(REPOSITORY_ID, getRepositoryId(options));
        licenseProperties.put(EXPIRES_AFTER, options.getExpiresAfter());
        licenseProperties.put(MAX_NUMBER_OF_USERS, options.getMaxNumberOfUsers());
        if(options.getMaxNumberOfCis() != null) {
            licenseProperties.putAll(MAX_NUMBER_OF_CIS, options.getMaxNumberOfCis());
        }
        if(options.getLicensedPlugin() != null) {
            licenseProperties.putAll(LICENSED_PLUGINS, options.getLicensedPlugin());
        }
        licenseProperties.put(SUPPORT_POLICY, options.getSupportPolicy());
        return Multimaps.filterEntries(licenseProperties, new Predicate<Map.Entry<LicenseProperty, String>>() {
            @Override
            public boolean apply(final Map.Entry<LicenseProperty, String> input) {
                return input.getValue() != null && !input.getValue().trim().isEmpty();
            }
                });
    }

    private static String getRepositoryId(LicenseGeneratorOptions options) {
        String repositoryId = options.getRepositoryId();
        if (repositoryId == null && options.isGenerateRepositoryId()) {
            repositoryId = UUID.randomUUID().toString();
        }
        return repositoryId;
    }
}
