package com.xebialabs.license.generator;

import java.io.PrintStream;
import java.util.List;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

import com.xebialabs.license.LicenseVersion1;
import com.xebialabs.license.LicenseVersion2;

/**
 */
public class LicenseGeneratorOptions {

    public enum Mode {
        GENERATE_LICENSE,
        GENERATE_KEYPAIR,
        VALIDATE_KEY
    }

    @Option(name = "-contact", usage = "The contact responsible for the license.", required = false)
    private String contact;

    @Option(name = "-licensedTo", usage = "The company the license will be for.", required = false)
    private String licensedTo;

    @Option(name = "-expiresAfter", usage = "The license will expire after this date yyyy-MM-dd", required = false)
    private String expiresAfter;

    @Option(name = "-product", usage = "The product the license is for. It is a required value. Possible values 'Deployit', 'XL Release', 'XL Deploy'.", required = false)
    private String product;

    @Option(name = "-generateRepositoryId", usage = "Generate a repository id.", required = false)
    private boolean generateRepositoryId;

    @Option(name = "-repositoryId", usage = "The repository id the license is for.", required = false)
    private String repositoryId;

    @Option(name = "-maxNumberOfUsers", usage = "The maximum number of users allowed by the license.", required = false)
    private String maxNumberOfUsers;

    @Option(name = "-f", usage = "The file to generate/check", required = false)
    private String file;

    @Option(name = "-s", usage = "The key size for a generated keypair (dev)", required = false)
    private int keySize;

    @Option(name = "-help", required = false, usage = "Prints this usage message")
    private boolean printUsage = false;

    @Option(name = "-k", usage = "Public/Private keypair", required = false)
    boolean generateKey;

    @Option(name = "-g", usage = "Generate license", required = false)
    boolean generateLicense;

    @Option(name = "-v", usage = "Validate license", required = false)
    boolean validateLicense;

    @Option(name = "-version", usage = "License version", required = false)
    private String licenseVersion;

    @Option(name = "-maxNumberOfCis", usage = "Licensed number of configuration items. This is a multi value property so you can use it like: `-maxNumberOfCis \"udm.Application=10\" -maxNumberOfCis \"overthere.Host=40\"`", required = false, multiValued = true)
    private List<String> maxNumberOfCis;

    @Option(name = "-licensedPlugin", usage = "Licensed plugin. This is a multi value property so you can use it like: `-licensedPlugin \"tomcat-plugin\" -licensedPlugin \"was-plugin\"`", required = false, multiValued = true)
    private List<String> licensedPlugin;

    @Option(name = "-supportPolicy", usage = "Support policy", required = false)
    private String supportPolicy;

    private Mode mode = Mode.GENERATE_LICENSE;

    public Mode getMode() {
        return mode;
    }

    public boolean isPrintUsage() {
        return printUsage;
    }

    public int getKeySize() {
        return keySize;
    }

    public String getFile() {
        return file;
    }

    public String getExpiresAfter() {
        return expiresAfter;
    }

    public String getProduct() {
        return product;
    }

    public String getContact() {
        return contact;
    }

    public String getLicensedTo() {
        return licensedTo;
    }

    public String getRepositoryId() {
        return repositoryId;
    }

    public String getMaxNumberOfUsers() {
        return maxNumberOfUsers;
    }

    public boolean isGenerateRepositoryId() {
        return generateRepositoryId;
    }

    public String getLicenseVersion() {
        return licenseVersion;
    }

    public List<String> getMaxNumberOfCis() {
        return maxNumberOfCis;
    }

    public List<String> getLicensedPlugin() {
        return licensedPlugin;
    }

    public String getSupportPolicy() {
        return supportPolicy;
    }

    public static LicenseGeneratorOptions parseCommandLine(String[] args) {
        LicenseGeneratorOptions options = new LicenseGeneratorOptions();
        final CmdLineParser parser = new CmdLineParser(options);
        try {
            parser.parseArgument(args);
            if (options.isPrintUsage()) {
                printUsage(parser, System.out);
                return null;
            }
            options.validate(parser);
        } catch (CmdLineException e) {
            printUsage(parser, System.err);
            System.err.println(e.getMessage());
            return null;
        }
        return options;
    }

    private static void printUsage(CmdLineParser parser, PrintStream stream) {
        System.out.println("license-generator arguments...");
        parser.printUsage(stream);
    }

    private void validate(CmdLineParser parser) throws CmdLineException {
        if (generateKey) {
            if (keySize != 1024 && keySize != 2048) {
                throw new CmdLineException(parser, "Key size must be 1024 or 2048.");
            }
            mode = Mode.GENERATE_KEYPAIR;
        } else if (generateLicense) {
            validateOptionIsNotEmpty(parser, product, "Product");
            validateOptionIsNotEmpty(parser, contact, "Contact");
            validateOptionIsNotEmpty(parser, licensedTo, "Licensed to");
            validateOptionIsNotEmpty(parser, file, "File ");
            if (licenseVersion != null && !licenseVersion.equals(LicenseVersion1.VERSION) && !licenseVersion.equals(LicenseVersion2.VERSION)) {
                throw new CmdLineException(parser, "Unsupported license version");
            }
            mode = Mode.GENERATE_LICENSE;
        } else if (validateLicense) {
            validateOptionIsNotEmpty(parser, file, "File ");
            mode = Mode.VALIDATE_KEY;
        } else {
            throw new CmdLineException(parser, "Please use one of the validate or generate modes.");
        }
    }

    private void validateOptionIsNotEmpty(final CmdLineParser parser, final String optionValue, final String optionName) throws CmdLineException {
        if (optionValue == null || optionValue.trim().isEmpty()) {
            throw new CmdLineException(parser, optionName + " cannot be empty");
        }
    }
}
