package com.xebialabs.license.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.UUID;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.HexEncoder;

import com.xebialabs.license.InvalidLicenseException;
import com.xebialabs.license.UnableToValidateLicenseException;
import com.xebialabs.license.v1.License;

import static com.xebialabs.license.LicenseProperty.CONTACT;
import static com.xebialabs.license.LicenseProperty.EXPIRES_AFTER;
import static com.xebialabs.license.LicenseProperty.LICENSED_TO;
import static com.xebialabs.license.LicenseProperty.MAX_NUMBER_OF_USERS;
import static com.xebialabs.license.LicenseProperty.PRODUCT;
import static com.xebialabs.license.LicenseProperty.REPOSITORY_ID;
import static com.xebialabs.license.LicenseReader.LICENSE_FILE_ENCODING;
import static com.xebialabs.license.LicenseReader.LICENSE_PREAMBLE;
import static com.xebialabs.license.LicenseReader.SIGNATURE_ENDING;
import static com.xebialabs.license.LicenseReader.SIGNATURE_PREAMBLE;
import static com.xebialabs.license.LicenseReader.readKey;
import static com.xebialabs.license.LicenseReader.readLicense;

public class LicenseGenerator {
    private final static String PRIV_KEY = "3082014b0201003082012c06072a8648ce3804013082011f02818100fd7f53811d75122952df4a9c2eece4e7f611b7523cef4400c31e3f80b6512669455d402251fb593d8d58fabfc5f5ba30f6cb9b556cd7813b801d346ff26660b76b9950a5a49f9fe8047b1022c24fbba9d7feb7c61bf83b57e7c6a8a6150f04fb83f6d3c51ec3023554135a169132f675f3ae2b61d72aeff22203199dd14801c70215009760508f15230bccb292b982a2eb840bf0581cf502818100f7e1a085d69b3ddecbbcab5c36b857b97994afbbfa3aea82f9574c0b3d0782675159578ebad4594fe67107108180b449167123e84c281613b7cf09328cc8a6e13c167a8b547c8d28e0a3ae1e2bb3a675916ea37f0bfa213562f1fb627a01243bcca4f1bea8519089a883dfe15ae59f06928b665e807b552564014c3bfecf492a041602140480cc8eb9735d2d83f0a5f4ffd15aa3c76d7298";

    private static void writeKey(Key key) throws IOException {
        byte[] bytes = key.getEncoded();
        HexEncoder hexit = new HexEncoder();
        hexit.encode(bytes, 0, bytes.length, System.out);
    }

    /** Utility to generate a new key pair. Mostly for testing. */
    private static void genKeyPair(int keySize) throws IOException, GeneralSecurityException {
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("DSA", "BC");
        SecureRandom random = SecureRandom.getInstance("SHA1PRNG", "BC");

        keyGen.initialize(keySize, random);
        KeyPair pair = keyGen.generateKeyPair();

        PrivateKey priv = pair.getPrivate();
        System.out.print("Private key: ");
        writeKey(priv);

        PublicKey pub = pair.getPublic();
        System.out.print("Public key : ");
        writeKey(pub);
    }

    private static void writeSigned(String content, String version, String outFile) throws IOException, GeneralSecurityException {
        OutputStream signedFile = new FileOutputStream(outFile);
        try {
            writeSigned(content, version, signedFile);
        } finally {
            signedFile.close();
        }
    }

    private static void writeSigned(String content, String version, OutputStream signedFile) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeySpecException, IOException, InvalidKeyException, SignatureException {
        String lineSeparator = System.getProperty("line.separator");

        KeyFactory kf = KeyFactory.getInstance("DSA", "BC");
        PrivateKey privateKey = kf.generatePrivate(new PKCS8EncodedKeySpec(readKey(PRIV_KEY)));

        Signature dsa = Signature.getInstance("SHA1withDSA", "BC");
        dsa.initSign(privateKey);

        byte[] data = content.getBytes(LICENSE_FILE_ENCODING);
        dsa.update(data);

        byte[] signature = dsa.sign();

        String preamble = String.format(LICENSE_PREAMBLE, version);
        signedFile.write(preamble.getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(data);

        signedFile.write(String.format(SIGNATURE_PREAMBLE, dsa.getAlgorithm()).getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));
        HexEncoder hexit = new HexEncoder();
        hexit.encode(signature, 0, signature.length, new LineBreakingOutputStream(signedFile));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));

        signedFile.write(SIGNATURE_ENDING.getBytes(LICENSE_FILE_ENCODING));
        signedFile.write(lineSeparator.getBytes(LICENSE_FILE_ENCODING));
    }

    public static boolean validateLicense(String licenseFilename) throws IOException {
        try {
            License license = readLicense(new File(licenseFilename));
            System.out.printf(license.toString());
            return true;
        } catch (InvalidLicenseException e) {
            System.out.println("License invalid: " + e.getMessage());
        } catch (UnableToValidateLicenseException e) {
            System.out.println("License check failed: " + e.getMessage());
        }
        return false;
    }

    public static void genLicense(String fileName, License license) throws IOException,
        GeneralSecurityException {
        writeSigned(license.toLicenseContent(), license.getLicenseVersion(), fileName);
    }
    
    public static void genLicense(OutputStream os, License license) throws IOException,
        GeneralSecurityException {
        writeSigned(license.toLicenseContent(), license.getLicenseVersion(), os);
    }

    public static License toLicense(String product, String licensedTo, String contact, String repositoryId, String expiresAfter, String maxNumberOfUsers) {
        License license = new License();
        if (product != null && !product.isEmpty()) {
            license.setProperty(PRODUCT, product);
        }
        license.setProperty(LICENSED_TO, licensedTo);
        license.setProperty(CONTACT, contact);
        license.setProperty(REPOSITORY_ID, repositoryId);
        license.setProperty(EXPIRES_AFTER, expiresAfter);
        license.setProperty(MAX_NUMBER_OF_USERS, maxNumberOfUsers);
        license.validate();
        return license;
    }

    public static void genLicense(String file, String product, String licensedTo, String contact, String repositoryId, String expiresAfter, String maxNumberOfUsers) throws IOException, GeneralSecurityException {
        License license = toLicense(product, licensedTo, contact, repositoryId, expiresAfter, maxNumberOfUsers);
        writeSigned(license.toLicenseContent(), license.getLicenseVersion(), file);
    }

    public static void main(String[] args) throws IOException, GeneralSecurityException {
        // ensure we have bouncy castle
        Security.addProvider(new BouncyCastleProvider());

        LicenseGeneratorOptions options = LicenseGeneratorOptions.parseCommandLine(args);
        if (options == null) {
            return;
        }

        switch (options.getMode()) {
        case GENERATE_KEYPAIR:
            genKeyPair(options.getKeySize());
            break;
        case GENERATE_LICENSE:
            genLicense(options.getFile(), options.getProduct(), options.getLicensedTo(), options.getContact(), getRepositoryId(options), options.getExpiresAfter(), options.getMaxNumberOfUsers());
            break;
        case VALIDATE_KEY:
            validateLicense(options.getFile());
            break;
        }
    }

    private static String getRepositoryId(LicenseGeneratorOptions options) {
        String repositoryId = options.getRepositoryId();
        if (repositoryId == null && options.isGenerateRepositoryId()) {
            repositoryId = UUID.randomUUID().toString();
        }
        return repositoryId;
    }
}
