package com.xebialabs.license.generator;

import java.io.PrintStream;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;

/**
 */
public class LicenseGeneratorOptions {
    public static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    public enum Mode {
        GENERATE_LICENSE,
        GENERATE_KEYPAIR,
        VALIDATE_KEY
    }

    @Option(name = "-contact", usage = "The contact responsible for the license.", required = false)
    private String contact;

    @Option(name = "-licensedTo", usage = "The company the license will be for.", required = false)
    private String licensedTo;

    @Option(name = "-expiresAfter", usage = "The license will expire after this date yyyy-MM-dd", required = false)
    private String expiresAfter;

    @Option(name = "-product", usage = "The product the license is for. It is a required value. Possible values 'Deployit', 'XL Release', 'XL Deploy'.", required = true)
    private String product;

    @Option(name = "-generateRepositoryId", usage = "Generate a repository id.", required = false)
    private boolean generateRepositoryId;

    @Option(name = "-repositoryId", usage = "The repository id the license is for.", required = false)
    private String repositoryId;

    @Option(name = "-f", usage = "The file to generate/check", required = false)
    private String file;

    @Option(name = "-s", usage = "The key size for a generated keypair (dev)", required = false)
    private int keySize;

    @Option(name = "-help", required = false, usage = "Prints this usage message")
    private boolean printUsage = false;

    @Option(name = "-k", usage = "Public/Private keypair", required = false)
    boolean generateKey;

    @Option(name = "-g", usage = "Generate license", required = false)
    boolean generateLicense;

    @Option(name = "-v", usage = "Validate license", required = false)
    boolean validateLicense;

    private Mode mode = Mode.GENERATE_LICENSE;

    public Mode getMode() {
        return mode;
    }

    public boolean isPrintUsage() {
        return printUsage;
    }

    public int getKeySize() {
        return keySize;
    }

    public String getFile() {
        return file;
    }

    public String getExpiresAfter() {
        return expiresAfter;
    }

    public String getProduct() {
        return product;
    }

    public String getContact() {
        return contact;
    }

    public String getLicensedTo() {
        return licensedTo;
    }

    public String getRepositoryId() {
        return repositoryId;
    }

    public boolean isGenerateRepositoryId() {
        return generateRepositoryId;
    }

    public static LicenseGeneratorOptions parseCommandLine(String[] args) {
        LicenseGeneratorOptions options = new LicenseGeneratorOptions();
        final CmdLineParser parser = new CmdLineParser(options);
        try {
            parser.parseArgument(args);
            if (options.isPrintUsage()) {
                printUsage(parser, System.out);
                return null;
            }
            options.validate(parser);
        } catch (CmdLineException e) {
            printUsage(parser, System.err);
            System.err.println(e.getMessage());
            return null;
        }
        return options;
    }

    private static void printUsage(CmdLineParser parser, PrintStream stream) {
        System.out.println("license-generator arguments...");
        parser.printUsage(stream);
    }

    private void validate(CmdLineParser parser) throws CmdLineException {
        if (generateKey) {
            if (keySize != 1024 && keySize != 2048) {
                throw new CmdLineException(parser, "Key size must be 1024 or 2048.");
            }
            mode = Mode.GENERATE_KEYPAIR;
        } else if (generateLicense) {
            if (contact == null || contact.trim().isEmpty()) {
                throw new CmdLineException(parser, "Contact cannot be empty");
            }
            if (licensedTo == null || licensedTo.trim().isEmpty()) {
                throw new CmdLineException(parser, "Licensed to cannot be empty");
            }
            if (expiresAfter == null || getExpiresAfter() == null) {
                throw new CmdLineException(parser, "Need an expiry date");
            }
            if (file == null || file.trim().isEmpty()) {
                throw new CmdLineException(parser, "Need a filename to generate");
            }
            mode = Mode.GENERATE_LICENSE;
        } else if (validateLicense) {
            if (file == null || file.trim().isEmpty()) {
                throw new CmdLineException(parser, "Need a filename to validate");
            }
            mode = Mode.VALIDATE_KEY;
        } else {
            throw new CmdLineException(parser, "Please use one of the validate or generate modes.");
        }
    }
}
