package com.xebialabs.license.generator;

import java.io.IOException;
import java.io.OutputStream;

/** Wrap {@link OutputStream} and insert line breaks while writing. Does not close the wrapped {@link OutputStream}. */
public class LineBreakingOutputStream extends OutputStream {
    private OutputStream wrappee;
    private int written;
    private final int maxWidth;
    private byte[] lineSeparator;

    /** Wrap {@link OutputStream} and insert line breaks every 80 characters. Does not close the wrapped {@link OutputStream}. */
    public LineBreakingOutputStream(OutputStream toWrap) {
        this(toWrap, 80, System.getProperty("line.separator"));
    }

    /** Wrap {@link OutputStream} and insert line breaks every maxWidth characters. Does not close the wrapped {@link OutputStream}. */
    public LineBreakingOutputStream(OutputStream toWrap, int maxWidth) {
        this(toWrap, maxWidth, System.getProperty("line.separator"));
    }

    /** Wrap {@link OutputStream} and insert lineSeparator every maxWidth characters. Does not close the wrapped {@link OutputStream}. */
    public LineBreakingOutputStream(OutputStream toWrap, int maxWidth, String lineSeparator) {
        this.wrappee = toWrap;
        this.maxWidth = maxWidth;
        this.lineSeparator = lineSeparator.getBytes();
        this.written = 0;
    }

    @Override
    public void write(int b) throws IOException {
        if (b == '\n' || b == '\r') {
            written = 0;
        }
        if (written >= maxWidth) {
            wrappee.write(lineSeparator);
            written = 0;
        }
        wrappee.write(b);
        written++;
    }

    @Override
    public void flush() throws IOException {
        wrappee.flush();
    }
}
