package com.xebialabs.license;

import java.util.Map;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.license.LicensePropertyType.DATE;
import static com.xebialabs.license.LicensePropertyType.LIST_OF_STRINGS;
import static com.xebialabs.license.LicensePropertyType.MAP_STRING_INTEGER;
import static com.xebialabs.license.LicensePropertyType.STRING;

/**
 * Property names used in license file
 */
public enum LicenseProperty {
    LICENSED_TO("Licensed to", true, STRING),
    CONTACT("Contact", true, STRING),
    EXPIRES_AFTER("Expires after", true, DATE),
    PRODUCT("Product", true, STRING),
    REPOSITORY_ID("Repository id", false, STRING),
    MAX_NUMBER_OF_USERS("Maximum number of users", false, STRING),
    MAX_NUMBER_OF_CIS("Licensed number of configuration items", false, MAP_STRING_INTEGER),
    LICENSED_PLUGINS("Licensed plugins", false, LIST_OF_STRINGS),
    SUPPORT_POLICY("Support policy", false, STRING);

    private static Map<String, LicenseProperty> lookup = newHashMap();
    static {
        for (LicenseProperty e : LicenseProperty.values()) {
            lookup.put(e.getName(), e);
        }
    }

    private final String name;
    private final boolean required;
    private final LicensePropertyType type;

    LicenseProperty(String name, boolean required, final LicensePropertyType type) {
        this.name = name;
        this.required = required;
        this.type = type;
    }

    public static LicenseProperty get(String name) {
        return lookup.get(name);
    }

    public static int getLongerNameLength() {
        int maxLength = 0;
        for (LicenseProperty licenseProperty : LicenseProperty.values()) {
            int length = licenseProperty.getName().length();
            if (length > maxLength) {
                maxLength = length;
            }
        }
        return maxLength;
    }

    public String getName() {
        return this.name;
    }

    public boolean isRequired() {
        return required;
    }

    public LicensePropertyType getType() {
        return type;
    }
}
