package com.xebialabs.license;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import static com.xebialabs.license.LicenseProperty.PRODUCT;

public abstract class License {

    public static final String PRODUCT_DEPLOYIT = "XL Deploy";
    public static final String PRODUCT_OLD_DEPLOYIT = "Deployit";
    public static final String PRODUCT_XL_RELEASE = "XL Release";

    public static final Set<String> PRODUCTS = Sets.newHashSet(PRODUCT_DEPLOYIT, PRODUCT_XL_RELEASE, PRODUCT_OLD_DEPLOYIT);

    static final DateTimeFormatter LICENSE_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    private Map<LicenseProperty, String> values;

    protected License(Map<LicenseProperty, String> values) {
        this.values = Maps.newHashMap(filterLicenseProperties(values));
        validateLicenseFormat();
    }

    public abstract List<LicenseProperty> getLicenseProperties();

    public abstract String getLicenseVersion();

    private Map<LicenseProperty, String> filterLicenseProperties(Map<LicenseProperty, String> values) {
        return Maps.filterEntries(values, new Predicate<Map.Entry<LicenseProperty, String>>() {
            @Override
            public boolean apply(Map.Entry<LicenseProperty, String> input) {
                return input.getValue() != null && !input.getValue().trim().isEmpty();
            }
        });
    }

    protected void validateProperties() {
        for (LicenseProperty property : getLicenseProperties()) {
            validateRequired(property);
            validateValueFormat(property);
        }
    }

    private void validateRequired(final LicenseProperty property) {
        if (property.isRequired() && !hasLicenseProperty(property)) {
            throw new InvalidLicenseException(emptyErrorMessage(property));
        }
    }

    private void validateProduct() {
        String product = getStringValue(PRODUCT);
        if(!PRODUCTS.contains(product)) {
            throw new InvalidLicenseException(String.format("product should be one of: %s got '%s'", PRODUCTS, product));
        }
    }

    private void validateValueFormat(LicenseProperty property) {
        switch (property.getType()) {
            case DATE:
                parseDateAndValidate(property);
                break;
            case MAP_STRING_INTEGER:
                parseMapOfStringIntegerAndValidate(property);
                break;
            case LIST_OF_STRINGS:
                parseListOfStrings(property);
                break;
        }
    }

    public void validateLicenseFormat(){
        validateProperties();
        validateProduct();
    }

    public boolean isDateExpired() {
        return LocalDate.now().isAfter(getLocalDateValue(LicenseProperty.EXPIRES_AFTER));
    }

    public String getStringValue(LicenseProperty key) {
        return values.get(key);
    }

    public LocalDate getLocalDateValue(LicenseProperty key) {
        return parseDateAndValidate(key);
    }

    public Map<String, Integer> getMapValue(LicenseProperty key) {
        return parseMapOfStringIntegerAndValidate(key);
    }

    public List<String> getListValue(LicenseProperty key) {
        return parseListOfStrings(key);
    }

    public boolean hasLicenseProperty(LicenseProperty key) {
        return values.containsKey(key);
    }


    private LocalDate parseDateAndValidate(LicenseProperty property) {
        String propertyValue = getStringValue(property);
        try {
            return LocalDate.parse(propertyValue, LICENSE_DATE_FORMAT);
        } catch (Exception e) {
            throw new InvalidLicenseException(createErrorMessage(property, propertyValue));
        }
    }

    private Map<String, Integer> parseMapOfStringIntegerAndValidate(LicenseProperty property) {
        String propertyValue = getStringValue(property);
        if (propertyValue == null) {
            return null;
        }
        Map<String, Integer> result = new HashMap<>();
        String[] pairs = propertyValue.split(",");
        for (String pair : pairs) {
            try {
                String[] pairStringInt = pair.trim().split("=");
                Integer intValue = Integer.parseInt(pairStringInt[1].trim());
                String stringValue = pairStringInt[0].trim();
                result.put(stringValue, intValue);
            } catch (Exception e) {
                throw new InvalidLicenseException(createErrorMessage(property, propertyValue));
            }
        }
        return result;
    }

    private List<String> parseListOfStrings(LicenseProperty property) {
        String propertyValue = getStringValue(property);
        if (propertyValue == null) {
            return null;
        }
        List<String> result = new ArrayList<>();
        String[] values = propertyValue.split(",");
        for (String val : values) {
            result.add(val.trim());
        }
        return result;
    }

    private String createErrorMessage(LicenseProperty property, String propertyValue) {
        return String.format("Unable to parse '%s' '%s'", property.getName().toLowerCase(), propertyValue);
    }

    private String emptyErrorMessage(LicenseProperty property) {
        return String.format("'%s' cannot be empty", property.getName().toLowerCase());
    }

    private String formatAll(String format) {
        return formatAllWithProperties(format, getLicenseProperties());
    }

    private String formatAllWithProperties(String template, List<LicenseProperty> properties) {
        StringBuilder result = new StringBuilder();
        for (LicenseProperty property : properties) {
            if (hasLicenseProperty(property)) {
                result.append(format(property, template));
            }
        }
        return result.toString();
    }

    public String toLicenseContent() {
        return formatAll("%s: %s%n");
    }

    @Override
    public String toString() {
        return formatAll("%-" + String.valueOf(LicenseProperty.getLongerNameLength()) + "s: %s%n");
    }


    protected String format(LicenseProperty licenseProperty, String template) {
        String value = getStringValue(licenseProperty);
        if (value != null) {
            return String.format(template, licenseProperty.getName(), value);
        }
        return null;
    }
}
