package com.xebialabs.license.v1;

import java.util.Set;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;

import com.xebialabs.license.InvalidLicenseException;
import com.xebialabs.license.LicenseProperty;

import static com.xebialabs.license.LicenseProperty.CONTACT;
import static com.xebialabs.license.LicenseProperty.EXPIRES_AFTER;
import static com.xebialabs.license.LicenseProperty.LICENSED_TO;
import static com.xebialabs.license.LicenseProperty.MAX_NUMBER_OF_USERS;
import static com.xebialabs.license.LicenseProperty.PRODUCT;
import static com.xebialabs.license.LicenseProperty.REPOSITORY_ID;

public class License {
    public static final String VERSION = "1";
    public static final String LICENSE_V1_PREAMBLE = "--- License V" + VERSION + " ---";

    public static final String PRODUCT_DEPLOYIT = "XL Deploy";
    public static final String PRODUCT_OLD_DEPLOYIT = "Deployit";
    public static final String PRODUCT_XL_RELEASE = "XL Release";
    public static final Set<String> PRODUCTS = Sets.newHashSet(PRODUCT_DEPLOYIT, PRODUCT_XL_RELEASE, PRODUCT_OLD_DEPLOYIT);

    public static final DateTimeFormatter LICENSE_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    protected LocalDate expiresAfter;
    protected String licensedTo;
    protected String contact;
    protected String repositoryId;
    protected String maxNumberOfUsers;
    protected String product;

    public void validate() {
        Preconditions.checkNotNull(getExpiresAfter(), "expiresAfter cannot be null");
        Preconditions.checkNotNull(licensedTo, "licensedTo cannot be null");
        Preconditions.checkArgument(!licensedTo.trim().isEmpty(), "licensedTo cannot be empty");
        Preconditions.checkNotNull(contact, "contact cannot be null");
        Preconditions.checkArgument(!contact.trim().isEmpty(), "contact cannot be empty");
        Preconditions.checkArgument(PRODUCTS.contains(getProduct()), String.format("product should be one of: %s got '%s'", PRODUCTS, getProduct()));
    }

    public String getLicenseVersion() {
        return VERSION;
    }

    public LocalDate getExpiresAfter() {
        return expiresAfter;
    }

    public String getLicensedTo() {
        return licensedTo;
    }

    public String getContact() {
        return contact;
    }

    public String getProduct() {
        return product;
    }

    public String getRepositoryId() {
        return repositoryId;
    }

    public String getMaxNumberOfUsers() {
        return maxNumberOfUsers;
    }

    public boolean setProperty(LicenseProperty licenseProperty, String value) {
        if (licenseProperty.isRequired()) {
            Preconditions.checkNotNull(value, String.format("Cannot set %s to null", licenseProperty.getName()));
        }
        value = value != null ? value.trim() : null;
        switch (licenseProperty) {
            case LICENSED_TO:           licensedTo = value;                 break;
            case CONTACT:               contact = value;                    break;
            case EXPIRES_AFTER:         expiresAfter = parseDate(value);    break;
            case PRODUCT:               product = value;                    break;
            case REPOSITORY_ID:         repositoryId = value;               break;
            case MAX_NUMBER_OF_USERS:   maxNumberOfUsers = value;           break;
            default: return false;
        }
        return true;
    }

    public Object getProperty(LicenseProperty licenseProperty) {
        switch (licenseProperty) {
            case LICENSED_TO:           return getLicensedTo();
            case CONTACT:               return getContact();
            case EXPIRES_AFTER:         return getExpiresAfter();
            case PRODUCT:               return getProduct();
            case REPOSITORY_ID:         return getRepositoryId();
            case MAX_NUMBER_OF_USERS:   return getMaxNumberOfUsers();
            default:
                throw new IllegalArgumentException(String.format("unknown property '%s'", licenseProperty.getName()));
        }
    }

    private String format(LicenseProperty licenseProperty, String format) {
        switch (licenseProperty) {
            case EXPIRES_AFTER:
                return String.format(format, EXPIRES_AFTER.getName(), LICENSE_DATE_FORMAT.print(getExpiresAfter()));
            default:
                String value = (String) getProperty(licenseProperty);
                return value != null ? String.format(format, licenseProperty.getName(), value) : "";
        }
    }

    private String formatAll(String format) {
        StringBuilder licenseContent = new StringBuilder();
        licenseContent.append(format(PRODUCT, format));
        licenseContent.append(format(LICENSED_TO, format));
        licenseContent.append(format(CONTACT, format));
        licenseContent.append(format(REPOSITORY_ID, format));
        licenseContent.append(format(EXPIRES_AFTER, format));
        licenseContent.append(format(MAX_NUMBER_OF_USERS, format));
        return licenseContent.toString();
    }

    private static LocalDate parseDate(String date) {
        try {
            return LocalDate.parse(date, LICENSE_DATE_FORMAT);
        } catch (Exception e) {
            throw new InvalidLicenseException(String.format("Unable to parse date '%s'", date));
        }
    }

    public String toLicenseContent() {
        return formatAll("%s: %s%n");
    }

    @Override
    public String toString() {
        return formatAll("%-" + String.valueOf(LicenseProperty.getLongerNameLength()) + "s: %s%n");
    }
}
