package com.xebialabs.license;

import java.util.*;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.license.LicenseProperty.PRODUCT;

public abstract class License {

    public static final String PRODUCT_DEPLOYIT = "XL Deploy";
    public static final String PRODUCT_OLD_DEPLOYIT = "Deployit";
    public static final String PRODUCT_XL_RELEASE = "XL Release";

    public static final Set<String> PRODUCTS = Sets.newHashSet(PRODUCT_DEPLOYIT, PRODUCT_XL_RELEASE, PRODUCT_OLD_DEPLOYIT);

    static final DateTimeFormatter LICENSE_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    private Multimap<LicenseProperty, String> values;

    protected License(Multimap<LicenseProperty, String> values) {
        this.values = ArrayListMultimap.create(filterLicenseProperties(values));
        validateLicenseFormat();
    }

    public abstract List<LicenseProperty> getLicenseProperties();

    public abstract String getLicenseVersion();

    private Multimap<LicenseProperty, String> filterLicenseProperties(Multimap<LicenseProperty, String> values) {
        return Multimaps.filterEntries(values, new Predicate<Map.Entry<LicenseProperty, String>>() {
            @Override
            public boolean apply(final Map.Entry<LicenseProperty, String> input) {
                return input.getValue() != null && !input.getValue().trim().isEmpty();
            }
        });
    }

    protected void validateProperties() {
        for (LicenseProperty property : getLicenseProperties()) {
            validateRequired(property);
            validateValueFormat(property);
        }
    }

    private void validateRequired(final LicenseProperty property) {
        if (property.isRequired() && !hasLicenseProperty(property)) {
            throw new InvalidLicenseException(emptyErrorMessage(property));
        }
    }

    private void validateProduct() {
        String product = getStringValue(PRODUCT);
        if(!PRODUCTS.contains(product)) {
            throw new InvalidLicenseException(String.format("product should be one of: %s got '%s'", PRODUCTS, product));
        }
    }

    private void validateValueFormat(LicenseProperty property) {
        switch (property.getType()) {
            case DATE:
                parseDateAndValidate(property);
                break;
            case MAP_STRING_INTEGER:
                parseMapOfStringIntegerAndValidate(property);
                break;
            case LIST_OF_STRINGS:
                getListValue(property);
                break;
        }
    }

    public void validateLicenseFormat(){
        validateProperties();
        validateProduct();
    }

    public boolean isDateExpired() {
        LocalDate localDateValue = getLocalDateValue(LicenseProperty.EXPIRES_AFTER);
        if(localDateValue == null){
            return false;
        } else {
            return LocalDate.now().isAfter(localDateValue);
        }
    }

    public String getStringValue(LicenseProperty key) {
        Collection<String> parts = values.get(key);
        if(parts.isEmpty()){
            return null;
        } else {
            return Joiner.on(",").skipNulls().join(parts);
        }
    }

    public LocalDate getLocalDateValue(LicenseProperty key) {
        return parseDateAndValidate(key);
    }

    public Map<String, Integer> getMapValue(LicenseProperty key) {
        return parseMapOfStringIntegerAndValidate(key);
    }

    public List<String> getListValue(LicenseProperty property) {
        return newArrayList(values.get(property));
    }

    public boolean hasLicenseProperty(LicenseProperty key) {
        return values.containsKey(key);
    }


    private LocalDate parseDateAndValidate(LicenseProperty property) {
        String propertyValue = getStringValue(property);
        if(!Strings.isNullOrEmpty(propertyValue)) {
            try {
                return LocalDate.parse(propertyValue, LICENSE_DATE_FORMAT);
            } catch (Exception e) {
                throw new InvalidLicenseException(createErrorMessage(property, propertyValue));
            }
        } else {
            return null;
        }
    }

    private Map<String, Integer> parseMapOfStringIntegerAndValidate(LicenseProperty property) {
        Collection<String> values = this.values.get(property);
        if (values == null) {
            return newHashMap();
        }
        Map<String, Integer> result = new HashMap<>();
        for (String pair : values) {
            try {
                String[] pairStringInt = pair.trim().split("=");
                Integer intValue = Integer.parseInt(pairStringInt[1].trim());
                String stringValue = pairStringInt[0].trim();
                result.put(stringValue, intValue);
            } catch (Exception e) {
                throw new InvalidLicenseException(createErrorMessage(property, values.toString()));
            }
        }
        return result;
    }

    private String createErrorMessage(LicenseProperty property, String propertyValue) {
        return String.format("Unable to parse '%s' '%s'", property.getName().toLowerCase(), propertyValue);
    }

    private String emptyErrorMessage(LicenseProperty property) {
        return String.format("'%s' cannot be empty", property.getName().toLowerCase());
    }

    private String formatAll(String format) {
        return formatAllWithProperties(format, getLicenseProperties());
    }

    private String formatAllWithProperties(String template, List<LicenseProperty> properties) {
        StringBuilder result = new StringBuilder();
        for (LicenseProperty property : properties) {
            if (hasLicenseProperty(property)) {
                String propertyString;
                if(property.getType() == LicensePropertyType.LIST_OF_STRINGS ||
                        property.getType() == LicensePropertyType.MAP_STRING_INTEGER){
                    propertyString = formatMultiValueProperty(property, template, ",");
                } else {
                    propertyString = formatStringProperty(property, template);
                }
                result.append(propertyString);
            }
        }
        return result.toString();
    }

    public String toLicenseContent() {
        return formatAll("%s: %s\n");
    }

    @Override
    public String toString() {
        return formatAll("%-" + String.valueOf(LicenseProperty.getLongerNameLength()) + "s: %s\n");
    }

    private String formatStringProperty(LicenseProperty licenseProperty, String template) {
        String value = getStringValue(licenseProperty);
        return String.format(template, licenseProperty.getName(), value);
    }

    private String formatMultiValueProperty(LicenseProperty licenseProperty, String template, String separator) {
        Collection<String> strings = values.get(licenseProperty);

        StringBuilder result = new StringBuilder();
        for(String value : strings){
            result.append(String.format(template, licenseProperty.getName(), value));
        }
        return result.toString();
    }
}
