package com.xebialabs.license.v1;

import java.util.Set;

import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;

import com.xebialabs.license.InvalidLicenseException;

public class License {
    public static final String VERSION = "1";
    public static final String LICENSE_V1_PREAMBLE = "--- License V" + VERSION + " ---";

    // property names used in license file
    public static final String LP_LICENSED_TO = "Licensed to";
    public static final String LP_CONTACT = "Contact";
    public static final String LP_EXPIRES_AFTER = "Expires after";
    public static final String LP_PRODUCT = "Product";
    public static final String LP_REPOSITORY_ID = "Repository id";

    public static final String PRODUCT_DEPLOYIT = "XL Deploy";
    public static final String PRODUCT_OLD_DEPLOYIT = "Deployit";
    public static final String PRODUCT_XL_RELEASE = "XL Release";
    public static final Set<String> PRODUCTS = Sets.newHashSet(PRODUCT_DEPLOYIT, PRODUCT_XL_RELEASE, PRODUCT_OLD_DEPLOYIT);

    public static final DateTimeFormatter LICENSE_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    protected LocalDate expiresAfter;
    protected String licensedTo;
    protected String contact;
    protected String repositoryId;
    protected String product;

    public void validate() {
        Preconditions.checkNotNull(getExpiresAfter(), "expiresAfter cannot be null");
        Preconditions.checkNotNull(licensedTo, "licensedTo cannot be null");
        Preconditions.checkArgument(!licensedTo.trim().isEmpty(), "licensedTo cannot be empty");
        Preconditions.checkNotNull(contact, "contact cannot be null");
        Preconditions.checkArgument(!contact.trim().isEmpty(), "contact cannot be empty");
        Preconditions.checkArgument(PRODUCTS.contains(getProduct()), String.format("product should be one of: %s got '%s'", PRODUCTS, getProduct()));
    }

    public String getLicenseVersion() {
        return VERSION;
    }

    public LocalDate getExpiresAfter() {
        return expiresAfter;
    }

    public String getLicensedTo() {
        return licensedTo;
    }

    public String getContact() {
        return contact;
    }

    public String getProduct() {
        return product;
    }

    public String getRepositoryId() {
        return repositoryId;
    }

    public boolean setProperty(String key, String value) {
        if (!LP_REPOSITORY_ID.equals(key)) {
            Preconditions.checkNotNull(value, String.format("Cannot set %s to null", key));
        }
        value = value != null ? value.trim() : null;
        if (LP_LICENSED_TO.equals(key)) {
            licensedTo = value;
        } else if (LP_CONTACT.equals(key)) {
            contact = value;
        } else if (LP_EXPIRES_AFTER.equals(key)) {
            expiresAfter = parseDate(value);
        } else if (LP_PRODUCT.equals(key)) {
            product = value;
        } else if (LP_REPOSITORY_ID.equals(key)) {
            repositoryId = value;
        } else {
            return false;
        }
        return true;
    }

    public Object getProperty(String key) {
        if (LP_LICENSED_TO.equals(key)) {
            return getLicensedTo();
        } else if (LP_CONTACT.equals(key)) {
            return getContact();
        } else if (LP_EXPIRES_AFTER.equals(key)) {
            return getExpiresAfter();
        } else if (LP_PRODUCT.equals(key)) {
            return getProduct();
        } else if (LP_REPOSITORY_ID.equals(key)) {
            return getRepositoryId();
        } else {
            throw new IllegalArgumentException(String.format("Unknown property '%s'", key));
        }
    }

    private static LocalDate parseDate(String date) {
        try {
            return LocalDate.parse(date, LICENSE_DATE_FORMAT);
        } catch (Exception e) {
            throw new InvalidLicenseException(String.format("Unable to parse date '%s'", date));
        }
    }

    public String toLicenseContent() {
        String productString = product == null ? "" : String.format("%s: %s%n", LP_PRODUCT, getProduct());
        String serverString = repositoryId == null ? "" : String.format("%s: %s%n", LP_REPOSITORY_ID, getRepositoryId());
        return String.format("%s%s: %s%n%s: %s%n%s%s: %s%n",
                productString,
                LP_LICENSED_TO, getLicensedTo(),
                LP_CONTACT, getContact(),
                serverString,
                LP_EXPIRES_AFTER, LICENSE_DATE_FORMAT.print(getExpiresAfter())
        );
    }

    @Override
    public String toString() {
        String serverString = repositoryId == null ? "" : String.format("Repository id : %s%n", getRepositoryId());
        return String.format(
            "Product       : %s%n" +
            "Licensed to   : %s%n" +
            "Contact       : %s%n" +
            "%s" +
            "Expires After : %s%n", getProduct(), getLicensedTo(), getContact(), serverString, LICENSE_DATE_FORMAT.print(getExpiresAfter())
            );
    }
}
