/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.jbossas.runbook;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.mapping.Mapping;
import com.xebialabs.deployit.mapper.artifact.*;
import com.xebialabs.deployit.plugin.jbossas.ci.JBossasDeploymentStrategy;
import com.xebialabs.deployit.plugin.jbossas.ci.JbossasArtifactServerMapping;
import com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer;
import com.xebialabs.deployit.plugin.jbossas.ci.JbossasStartMode;
import com.xebialabs.deployit.plugin.jbossas.mapper.*;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasRestartServerStep;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasStartServerStep;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasStopServerStep;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasWaitForDeploymentsToCompleteStep;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Supports the deployment, re-deployment and undeployment of an {@link com.xebialabs.deployit.ci.DeploymentPackage Application Package} to a
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}.
 * <p/>
 * <h4>Conditions</h4>
 * <p/>
 * Will trigger if the change plan contains the addition, modification or deletion of a {@link Deployment Deployment} CI to an
 * {@link com.xebialabs.deployit.ci.Environment Environment} CI which contains JBoss middleware CI's.
 * <p/>
 * Note that Additions, Modifications and Deletions are always handled in one single flow.
 * <p/>
 * <h4>Actions</h4>
 * <p/>
 * <ol>
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.Ear EARs} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.War WARs} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Destroy {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasResource JBoss resources} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.Ear EARs} to {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.War WARs} to {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Create {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasResource JBoss resources} from
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.Libraries} from {@link com.xebialabs.deployit.ci.Host}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} from {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.Libraries} on {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} on {@link com.xebialabs.deployit.ci.Host}
 * </ol>
 */
public class JbossasDeploymentRunBook extends SingleTypeHandlingRunBook<Deployment> {

	public JbossasDeploymentRunBook() {
		super(Deployment.class);
	}

	@Override
	protected void resolve(Change<Deployment> change, ChangePlan changePlan, List<Step> steps) {
		EjbJarToJbossasServerMapper ejbJarToServerMapper = new EjbJarToJbossasServerMapper(change);
		WarToJbossasServerMapper warToServerMapper = new WarToJbossasServerMapper(change);
		EarToJbossasServerMapper earToServerMapper = new EarToJbossasServerMapper(change);
		JbossasDataSourceToJbossasServerMapper dataSourceToServerMapper = new JbossasDataSourceToJbossasServerMapper(change);
		JbossasQueueToJbossasServerMapper queueToServerMapper = new JbossasQueueToJbossasServerMapper(change);
		JbossasTopicToJbossasServerMapper topicToServerMapper = new JbossasTopicToJbossasServerMapper(change);
		LibrariesToHostMapper librariesMapper = new LibrariesToHostMapper(change);
		GenericFolderToHostMapper genericFolderToHostMapper = new GenericFolderToHostMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);
		SqlFolderToDatabaseMapper sqlFoldermapper = new SqlFolderToDatabaseMapper(change);
		SqlScriptToDatabaseMapper sqlScriptMapper = new SqlScriptToDatabaseMapper(change);

		Set<JbossasServer> affectedServers = new HashSet<JbossasServer>();
		affectedServers.addAll(ejbJarToServerMapper.getAffectedTargets());
		affectedServers.addAll(warToServerMapper.getAffectedTargets());
		affectedServers.addAll(earToServerMapper.getAffectedTargets());
		affectedServers.addAll(dataSourceToServerMapper.getAffectedTargets());
		affectedServers.addAll(queueToServerMapper.getAffectedTargets());
		affectedServers.addAll(topicToServerMapper.getAffectedTargets());

		if (!isJbossasDeployment(ejbJarToServerMapper, warToServerMapper, earToServerMapper, dataSourceToServerMapper, queueToServerMapper, topicToServerMapper)) {
			return;
		}

		stopServers(affectedServers, steps);

		int lastStepCount = steps.size();
		earToServerMapper.generateDeletionSteps(steps);
		warToServerMapper.generateDeletionSteps(steps);
		ejbJarToServerMapper.generateDeletionSteps(steps);
		topicToServerMapper.generateDeletionSteps(steps);
		queueToServerMapper.generateDeletionSteps(steps);
		dataSourceToServerMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateDeletionSteps(steps);
		librariesMapper.generateDeletionSteps(steps);
		genericFolderToHostMapper.generateDeletionSteps(steps);

		// no modification steps as they don't exist for JBoss

		sqlFoldermapper.generateAdditionSteps(steps);
		sqlScriptMapper.generateAdditionSteps(steps);

		genericFolderToHostMapper.generateAdditionSteps(steps);
		librariesMapper.generateAdditionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);
		dataSourceToServerMapper.generateAdditionSteps(steps);
		queueToServerMapper.generateAdditionSteps(steps);
		topicToServerMapper.generateAdditionSteps(steps);
		ejbJarToServerMapper.generateAdditionSteps(steps);
		warToServerMapper.generateAdditionSteps(steps);
		earToServerMapper.generateAdditionSteps(steps);

		if (steps.size() > lastStepCount) {
			startServers(getTargets(change), steps);
			waitForDeployments(affectedServers, steps);
		}
	}

	private Set<JbossasServer> getTargets(Change<Deployment> change) {
		final Set<JbossasServer> servers = Sets.newTreeSet(new Comparator<JbossasServer>() {
			public int compare(JbossasServer wlsTarget, JbossasServer wlsTarget1) {
				return wlsTarget.getName().compareTo(wlsTarget1.getName());
			}
		});
		getTargets(servers, change.getNewRevision());
		getTargets(servers, change.getOldRevision());
		return servers;

	}

	private void getTargets(Set<JbossasServer> targets, Deployment deployment) {
		if (deployment == null)
			return;

		for (Mapping<?,?> m : deployment.getMappings()) {
			if (m instanceof JbossasArtifactServerMapping) {
				JbossasArtifactServerMapping mapping = (JbossasArtifactServerMapping) m;
				if (mapping.getDeploymentStrategy().equals(JBossasDeploymentStrategy.DISABLED_HOT_DEPLOYMENT_STRATEGY)) {
					targets.add(mapping.getTarget());
				}
			}
		}
	}

	private void stopServers(Set<JbossasServer> affectedServers, List<Step> steps) {
		if (affectedServers.isEmpty())
			return;

		for (JbossasServer server : affectedServers) {
			if (JbossasStartMode.STOP_START.equals(server.getStartMode())) {
				steps.add(new JbossasStopServerStep(server));
			}
		}
	}

	private boolean isJbossasDeployment(EjbJarToJbossasServerMapper ejbJarToServerMapper, WarToJbossasServerMapper warToServerMapper,
	                                    EarToJbossasServerMapper earToServerMapper, JbossasDataSourceToJbossasServerMapper dataSourceToServerMapper,
	                                    JbossasQueueToJbossasServerMapper queueToServerMapper, JbossasTopicToJbossasServerMapper topicToServerMapper) {
		Set<JbossasServer> allServers = new HashSet<JbossasServer>();
		allServers.addAll(ejbJarToServerMapper.getAllTargets());
		allServers.addAll(warToServerMapper.getAllTargets());
		allServers.addAll(earToServerMapper.getAllTargets());
		allServers.addAll(dataSourceToServerMapper.getAllTargets());
		allServers.addAll(queueToServerMapper.getAllTargets());
		allServers.addAll(topicToServerMapper.getAllTargets());
		return !allServers.isEmpty();
	}

	private void startServers(Set<JbossasServer> servers, List<Step> steps) {
		if (servers.isEmpty())
			return;
		for (JbossasServer eachServer : servers) {
			switch (eachServer.getStartMode()) {
				case RESTART:
					addRestartCommand(eachServer, steps);
					break;
				case STOP_START:
					steps.add(new JbossasStartServerStep(eachServer));
					break;
			}
		}
	}

	private void waitForDeployments(Set<JbossasServer> servers, List<Step> steps) {
		int deploymentCompletionWaitTime = 0;
		for (JbossasServer eachServer : servers) {
			final int completionWaitTime = eachServer.getDeploymentCompletionWaitTime();
			if (completionWaitTime > deploymentCompletionWaitTime)
				deploymentCompletionWaitTime = completionWaitTime;
		}
		if (deploymentCompletionWaitTime > 0)
			steps.add(new JbossasWaitForDeploymentsToCompleteStep(deploymentCompletionWaitTime));
	}

	protected void addRestartCommand(JbossasServer server, List<Step> steps) {
		switch (server.getVersion()) {
			case JBOSSAS_40:
				steps.add(new JbossasRestartServerStep(server));
				break;
			case JBOSSAS_50:
			case JBOSSAS_60:
				steps.add(new JbossasStopServerStep(server));
				steps.add(new JbossasWaitForDeploymentsToCompleteStep(server.getDeploymentCompletionWaitTime()));
				steps.add(new JbossasStartServerStep(server));
				break;
		}
	}

}
