/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.jbossas.ci;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.*;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.ListenServer;
import com.xebialabs.deployit.ci.LoadBalancedServerAware;
import com.xebialabs.deployit.ci.LoadBalancingProtocol;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.jbossas.step.CheckAjpPortStep;
import com.xebialabs.deployit.plugin.jbossas.step.CheckJbossasDeployDirPath;
import com.xebialabs.deployit.plugin.jbossas.step.CheckJbossasServerVersionStep;
import com.xebialabs.deployit.plugin.jbossas.step.CheckNamingPortStep;
import com.xebialabs.deployit.step.CheckDirExistsStep;
import com.xebialabs.deployit.step.CheckFileExistenceStep;
import com.xebialabs.deployit.step.HostConnectionStep;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * A JBoss Application Server
 */
@SuppressWarnings("serial")
@ConfigurationItem(description = "JBoss Application Server instance", discoverable = true, category = "middleware")
public class JbossasServer extends BaseConfigurationItem implements Discoverable<JbossasServer>, LoadBalancedServerAware {

	private static final int DEPLOYMENT_COMPLETION_WAIT_TIME_MILLIS = 10000;

	public static final ArrayList<LoadBalancingProtocol> SUPPORTED_PROTOCOLS = Lists.newArrayList(LoadBalancingProtocol.AJP);

	@ConfigurationItemProperty(required = true, label = "JBoss version", description = "Version of the JBoss server.")
	private JbossasVersion version = JbossasVersion.JBOSSAS_UNKNOWN;

	@ConfigurationItemProperty(required = true, asContainment = true, label = "JBoss server host", description = "Host on which the JBoss server is running.")
	private Host host;

	@ConfigurationItemProperty(required = true, discoveryParam = true, size = Size.LARGE, label = "JBoss installation location", description = "Place where JBoss is installed such as /usr/local/jboss-5.1.0.GA.")
	private String home;

	@ConfigurationItemProperty(required = true, discoveryParam = true, label = "JBoss instance name", description = "Name of the JBoss server instance, e.g. default or minimal.")
	private String name;

	@ConfigurationItemProperty(required = true, size = Size.LARGE, label = "JBoss start command", description = "Path to the script that should be executed to start the JBoss server in the background")
	private String startCommand;

	@ConfigurationItemProperty(required = false, size = Size.LARGE, label = "JBoss stop command", description = "Command that should be executed to stop the JBoss server.")
	private String stopCommand;

	@ConfigurationItemProperty(required = false, size = Size.LARGE, label = "JBoss restart command", description = "Command that should be executed to restart the JBoss server.")
	private String restartCommand;

	@ConfigurationItemProperty(required = false)
	private JbossasStartMode strategy = JbossasStartMode.RESTART;

	@ConfigurationItemProperty(required = false, description = "Estimated time in miliseconds to wait for the deployment completion.")
	private int deploymentCompletionWaitTime = DEPLOYMENT_COMPLETION_WAIT_TIME_MILLIS;

	@ConfigurationItemProperty(required = false, description = "AJP Port for the JBoss Server, default is 8009")
	private int ajpPort = 8009;

	@ConfigurationItemProperty(required = false, description = "ControlPort of the JBoss Server, default is 1099 for JBoss 5, 1090 for JBoss 6+")
	private int controlPort = 1099;

	@ConfigurationItemProperty
	private List<JbossasPath> deployDirectories = Lists.newArrayList();

	public HostSession connectToAdminHost() {
		return getHost().getHostSession();
	}

	public String getRunJarFilePath() {
		String fileSep = getHost().getFileSeparator();
		return getHome() + fileSep + "bin" + fileSep + "run.jar";
	}

	public String getInstanceDirPath() {
		String fileSep = getHost().getFileSeparator();
		return getHome() + fileSep + "server" + fileSep + getName();
	}

	public String getDeployDirPath() {
		String fileSep = getHost().getFileSeparator();
		// get the first configured deployed directory else "deploy"
		final String directory = (deployDirectories.isEmpty() ? "deploy" : deployDirectories.iterator().next().getName());
		return getInstanceDirPath() + fileSep + directory;
	}
	
	public String getShutdownCommand() {
		String fileSep = getHost().getFileSeparator();
		return getHome() + fileSep + "bin" + fileSep + "shutdown" + getHost().getOperatingSystemFamily().getScriptExtension();
	}

	public JbossasVersion getVersion() {
		return version;
	}

	public void setVersion(JbossasVersion version) {
		this.version = version;
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public String getHome() {
		return home;
	}

	public void setHome(String home) {
		this.home = home;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getRestartCommand() {
		return restartCommand;
	}

	public void setRestartCommand(String restartCommand) {
		this.restartCommand = restartCommand;
	}

	public String getTmpDirPath() {
		String fileSep = getHost().getFileSeparator();
		return getInstanceDirPath() + fileSep + "tmp";
	}

	public int getAjpPort() {
		return ajpPort;
	}

	public void setAjpPort(int ajpPort) {
		this.ajpPort = ajpPort;
	}

	public int getControlPort() {
		return controlPort;
	}

	public void setControlPort(int controlPort) {
		this.controlPort = controlPort;
	}

	public String getStartCommand() {
		return startCommand;
	}

	public void setStartCommand(String startCommand) {
		this.startCommand = startCommand;
	}

	public String getStopCommand() {
		return stopCommand;
	}

	public void setStopCommand(String stopCommand) {
		this.stopCommand = stopCommand;
	}

	public int getDeploymentCompletionWaitTime() {
		return deploymentCompletionWaitTime;
	}

	public void setDeploymentCompletionWaitTime(int deploymentCompletionWaitTime) {
		this.deploymentCompletionWaitTime = deploymentCompletionWaitTime;
	}

	public JbossasStartMode getStartMode() {
		return strategy;
	}

	public void setStartMode(JbossasStartMode strategy) {
		this.strategy = strategy;
	}

	public List<Step> getCheckSteps() {
		List<Step> steps = new ArrayList<Step>();
		steps.add(new HostConnectionStep(getHost()));
		steps.add(new CheckDirExistsStep(getHost(), getHome()));
		steps.add(new CheckDirExistsStep(getHost(), getInstanceDirPath()));
		steps.add(new CheckDirExistsStep(getHost(), getDeployDirPath()));
		steps.add(new CheckDirExistsStep(getHost(), getTmpDirPath()));
		steps.add(new CheckJbossasServerVersionStep(this));
		if (StringUtils.isNotEmpty(getRestartCommand())) {
			steps.add(new CheckFileExistenceStep(getHost(), getRestartCommand(), true, false, true));
		}
		steps.add(new CheckJbossasDeployDirPath(this));
		steps.add(new CheckNamingPortStep(this));
		steps.add(new CheckAjpPortStep(this));
		return steps;
	}

	public JbossasServer discover(Map<String, Object> info, ChangePlan cp) {
		List<Step> steps = getCheckSteps();
		cp.addSteps(steps);
		cp.execute(info);
		return this;
	}

	public JbossasServer discoverChildrenInfo(Map<String, Object> info, ChangePlan cp) {
		// no children to discover here
		return null;
	}

	public String getTwiddleFilePath() {
		String fileSep = getHost().getFileSeparator();
		return getHome() + fileSep + "bin" + fileSep + "twiddle" + getHost().getScriptExtension();
	}

	public void addDeployDir(String path) {
		deployDirectories.add(new JbossasPath(path));
	}

	public List<LoadBalancingProtocol> getSupportedProtocols() {
		return SUPPORTED_PROTOCOLS;
	}

	public List<ListenServer> getLoadBalancedServers(LoadBalancingProtocol protocol) {
		switch (protocol) {
		case AJP:
			final ListenServer server = new ListenServer();
			server.setLabel(getLabel());
			server.setHost(getHost());
			server.setListenPort(getAjpPort());
			return Lists.newArrayList(server);
		default:
			return Lists.newArrayList();
		}
	}
}
