package com.xebialabs.deployit.plugin.jbossas.runbook;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.mapper.artifact.ConfigurationFilesToHostMapper;
import com.xebialabs.deployit.mapper.artifact.LibrariesToHostMapper;
import com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer;
import com.xebialabs.deployit.plugin.jbossas.mapper.EarToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.JbossasQueueToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.JbossasDataSourceToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.JbossasTopicToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.mapper.WarToJbossasServerMapper;
import com.xebialabs.deployit.plugin.jbossas.step.JbossasRestartServerStep;

/**
 * Supports the deployment, re-deployment and undeployment of an 
 * {@link com.xebialabs.deployit.ci.DeploymentPackage Application Package} to a 
 * {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or
 * deletion of a {@link Deployment Deployment} CI to an {@link com.xebialabs.deployit.ci.Environment Environment} CI 
 * which contains JBoss middleware CI's.
 * 
 * Note that Additions, Modifications and Deletions are always handled in one single flow.
 * 
 * <h4>Actions</h4>
 * 
 * <ol>
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.plugins.was.runbooks.Ear ear} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.plugins.was.runbooks.War war} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Destroy {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasResource JBoss resources} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.plugins.was.runbooks.Ear ear} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.plugins.was.runbooks.War war} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Create {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasResource JBoss resources} from {@link com.xebialabs.deployit.plugin.jbossas.ci.JbossasServer JBoss Application Server}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.Libraries} from {@link com.xebialabs.deployit.ci.Host}
 * <li>Undeploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} from {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.Libraries} on {@link com.xebialabs.deployit.ci.Host}
 * <li>Deploy {@link com.xebialabs.deployit.ci.artifact.ConfigurationFiles} on {@link com.xebialabs.deployit.ci.Host}
 * </ol>
 * 
 */
public class JbossasDeploymentRunBook extends JbossasSingleTypeRunBook<Deployment> {

	public JbossasDeploymentRunBook() {
		super(Deployment.class);
	}

	protected void resolve(Change<Deployment> change, List<Step> steps) {
		WarToJbossasServerMapper warToServerMapper = new WarToJbossasServerMapper(change);
		EarToJbossasServerMapper earToServerMapper = new EarToJbossasServerMapper(change);
		JbossasDataSourceToJbossasServerMapper dataSourceToServerMapper = new JbossasDataSourceToJbossasServerMapper(change);
		JbossasQueueToJbossasServerMapper queueToServerMapper = new JbossasQueueToJbossasServerMapper(change);
		JbossasTopicToJbossasServerMapper topicToServerMapper = new JbossasTopicToJbossasServerMapper(change);
		LibrariesToHostMapper librariesMapper = new LibrariesToHostMapper(change);
		ConfigurationFilesToHostMapper configurationFilesToHostMapper = new ConfigurationFilesToHostMapper(change);

		Set<JbossasServer> allTargetServers = new HashSet<JbossasServer>();
		allTargetServers.addAll(warToServerMapper.getAllTargets());
		allTargetServers.addAll(earToServerMapper.getAllTargets());
		allTargetServers.addAll(dataSourceToServerMapper.getAllTargets());
		allTargetServers.addAll(queueToServerMapper.getAllTargets());
		allTargetServers.addAll(topicToServerMapper.getAllTargets());
		
		Set<Host> targetHosts = new HashSet<Host>();
		targetHosts.addAll(configurationFilesToHostMapper.getAllTargets());
		targetHosts.addAll(librariesMapper.getAllTargets());

		if (allTargetServers.isEmpty() && targetHosts.isEmpty()) {
			return;
		}

		int lastStepCount = steps.size();
		earToServerMapper.generateDeletionSteps(steps);
		warToServerMapper.generateDeletionSteps(steps);
		dataSourceToServerMapper.generateDeletionSteps(steps);
		queueToServerMapper.generateDeletionSteps(steps);
		topicToServerMapper.generateDeletionSteps(steps);
		// no modification steps as they don't exist for JBoss
		earToServerMapper.generateAdditionSteps(steps);
		warToServerMapper.generateAdditionSteps(steps);
		dataSourceToServerMapper.generateAdditionSteps(steps);
		queueToServerMapper.generateAdditionSteps(steps);
		topicToServerMapper.generateAdditionSteps(steps);
		librariesMapper.generateDeletionSteps(steps);
		librariesMapper.generateAdditionSteps(steps);
		configurationFilesToHostMapper.generateDeletionSteps(steps);
		configurationFilesToHostMapper.generateAdditionSteps(steps);

		if (steps.size() > lastStepCount) {
			restartServers(allTargetServers, steps);
		}
	}

	private void restartServers(Set<JbossasServer> servers, List<Step> steps) {
		for (JbossasServer eachServer : servers) {
			steps.add(new JbossasRestartServerStep(eachServer));
		}
	}

}
