package com.xebialabs.deployit.test.repository;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import static com.xebialabs.platform.test.TestUtils.newInstance;

public
class InMemoryRepository implements Repository {

    public static final AtomicReference<InMemoryRepository> REFERENCE = new AtomicReference<>(new InMemoryRepository());

    private Map<String, ConfigurationItem> store = new HashMap<>();

    public void initializeWithFakeRootFolders() {
        create(newInstance("test.Root", "Configuration"));
        create(newInstance("test.Root", "Infrastructure"));
        create(newInstance("test.Root", "Applications"));
        create(newInstance("test.Root", "Environments"));
    }

    @Override
    public boolean exists(String id) {
        return store.containsKey(id);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> T read(String id) {
        if (!exists(id)) {
            throw new IllegalArgumentException("Repository does not contain item with id " + id);
        }
        return (T) store.get(id);
    }

    @Override
    public <T extends ConfigurationItem> List<T> read(List<String> ids, Integer depth) {
        return ids.stream().map(this::<T>read).collect(Collectors.toList());
    }

    @SafeVarargs
    @Override

    public final <T extends ConfigurationItem> void create(T... entities) {
        checkThatEntitiesDoNotExist(entities);
        storeEntities(entities);
    }

    @SafeVarargs
    @Override
    public final <T extends ConfigurationItem> void update(T... entities) {
        storeEntities(entities);
    }

    @SafeVarargs
    @Override
    public final <T extends ConfigurationItem> void createOrUpdate(T... entities) {
        storeEntities(entities);
    }

    @Override
    public void delete(String... ids) {
        for (String id : ids) {
            store.remove(id);
        }
    }

    @Override
    public void move(String id, String newId) {
        ConfigurationItem ci = store.remove(id);
        ci.setId(newId);
        store.put(newId, ci);
    }

    @Override
    public void rename(String id, String newName) {
        int i = id.lastIndexOf("/");
        String newId = newName;
        if (i > -1) {
            newId = id.substring(0, i + 1) + newName;
        }
        move(id, newId);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> List<T> search(Type type) {
        List<T> entities = new ArrayList<>();
        for (ConfigurationItem configurationItem : store.values()) {
            if (type == null || configurationItem.getType().instanceOf(type)) {
                entities.add((T) configurationItem);
            }
        }
        return entities;
    }

    @Override
    public <T extends ConfigurationItem> List<T> search(Type type, final String parent) {
        List<T> search = this.search(type);
        return search.stream().filter(ci -> ci.getId().startsWith(parent)).collect(Collectors.toList());
    }

    private void checkThatEntitiesDoNotExist(final ConfigurationItem[] entities) {
        for (ConfigurationItem entity : entities) {
            if (store.containsKey(entity.getId())) {
                throw new IllegalStateException("Entity " + entity.getId() + " already exists and cannot be created twice.");
            }
        }
    }

    private void storeEntities(final ConfigurationItem[] entities) {
        for (ConfigurationItem entity : entities) {
            store.put(entity.getId(), entity);
        }
    }

    public void clear() {
        store.clear();
    }
}
