package com.xebialabs.platform.test;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.Joiner;
import com.google.common.io.Files;
import com.google.common.io.Resources;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.overthere.local.LocalFile;

import static com.google.common.collect.Sets.newHashSet;

public class TestUtils {

    /**
     * Returns resource content as a string
     */
    public static String readResource(String resource) throws IOException {
        return Resources.toString(Resources.getResource(resource), Charset.defaultCharset());
    }

    @SuppressWarnings("unchecked")
    public static <T> T newInstance(String type) {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(type);
        if (descriptor == null) {
            throw new RuntimeException("Cannot instantiate unknown type");

        }
        return (T) descriptor.newInstance();
    }

    public static <T> T newInstance(Class<T> clazz) {
        return newInstance(Type.valueOf(clazz).toString());
    }

    @SuppressWarnings("unchecked")
    public static <T> T newInstance(Type type, String id) {
        return (T) type.getDescriptor().newInstance(id);
    }

    /**
     * Creates new CI instance and sets id
     */
    public static <T extends ConfigurationItem> T newInstance(Class<T> clazz, String id) {
        @SuppressWarnings("unchecked")
        T instance = (T) newInstance(Type.valueOf(clazz).toString());
        instance.setId(id);
        return instance;
    }

    /**
     * Creates new CI instance and sets id
     */
    public static <T extends ConfigurationItem> T newInstance(String type, String id) {
        T instance = newInstance(type);
        instance.setId(id);
        return instance;
    }

    public static String id(String... names) {
        return Joiner.on('/').join(names);
    }

    public static DeploymentPackage createDeploymentPackage(Deployable... deployables) {
        return createDeploymentPackage("1.0", deployables);
    }

    public static DeploymentPackage createDeploymentPackage(String version, Deployable... deployables) {
        Application app = newInstance(Application.class);
        app.setId(id("Applications", "Test"));
        DeploymentPackage pkg = newInstance(DeploymentPackage.class);
        pkg.setId(id(pkg.getId(), version));
        pkg.setApplication(app);

        for (Deployable deployable : deployables) {
            deployable.setId(id(app.getId(), deployable.getId()));
            pkg.addDeployable(deployable);
        }

        return pkg;
    }

    public static Environment createEnvironment(Container... containers) {
        Environment env = newInstance(Environment.class);
        env.setId(id("Environments", "JUnit"));
        Set<Container> containerSet = newHashSet();
        containerSet.addAll(Arrays.asList(containers));
        env.setMembers(containerSet);
        return env;
    }

    public static <T extends SourceArtifact> T createArtifact(String name, String version, String classpathResource, String type, File workingFolder)
            throws IOException {
        T artifact = newInstance(type);
        artifact.setId(id("Applications", "Test", version, name));
        URL artifactURL = Thread.currentThread().getContextClassLoader().getResource(classpathResource);
        File artifactFile = new File(workingFolder, artifact.getName());
        artifactFile.createNewFile();
        Resources.asByteSource(artifactURL).copyTo(Files.asByteSink(artifactFile));
        artifact.setFile(LocalFile.valueOf(artifactFile));
        return artifact;
    }

    public static <T extends SourceArtifact> T createArtifactWithoutCopy(String name, String version, String classpathResource, String type, File workingFolder)
            throws URISyntaxException {
        @SuppressWarnings("unchecked")
        T artifact = (T) newInstance(type);
        artifact.setId(id("Applications", "Test", version, name));
        URL artifactURL = Thread.currentThread().getContextClassLoader().getResource(classpathResource);
        artifact.setFile(LocalFile.valueOf(new File(artifactURL.toURI().getPath())));
        return artifact;
    }

    public static String getTitleFromHtmlPage(String url) throws IOException {
        return getTitleFromHtmlPage(new URL(url));
    }
    public static String getTitleFromHtmlPage(URL url) throws IOException {
        String htmlContent = getHtmlContentFromUrl(url);
        String title = "";
        Pattern p = Pattern.compile("<title>(.*?)</title>");
        Matcher m = p.matcher(htmlContent);
        while (m.find() == true) {
            title = m.group(1);
        }
        return title;
    }

    private static String getHtmlContentFromUrl(URL url) throws IOException {
        URLConnection urlConnection = url.openConnection();
        InputStreamReader is = new InputStreamReader(urlConnection.getInputStream());
        BufferedReader dis = new BufferedReader(is);
        String html = "", tmp = "";
        while ((tmp = dis.readLine()) != null) {
            html += " " + tmp;
        }
        html = html.replaceAll("\\s+", " ");
        dis.close();
        return html;
    }
}
