package com.xebialabs.deployit.test.deployment;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.deployment.planner.*;
import com.xebialabs.deployit.deployment.rules.RuleBasedPlannerFactory;
import com.xebialabs.deployit.deployment.stager.DeploymentStager;
import com.xebialabs.deployit.engine.replacer.Placeholders;
import com.xebialabs.deployit.inspection.Inspector;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.test.support.TestExecutionContext;
import com.xebialabs.deployit.test.support.TestInspectionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.flow.StepExitCode.FAIL;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static java.lang.String.format;

public class DeployitTester {

    private final Planner planner;
    private RuleBasedPlannerFactory factory;
    private final Inspector inspector;
    private final com.xebialabs.deployit.test.repository.InMemoryRepository repository;
    private Function<String, Object> idToCi = new Function<String, Object>() {
        @Override
        public Object apply(String input) {
            return repository.read(input);
        }
    };

    private static final Splitter STRING_TO_COLLECTION_SPLITTER = Splitter.on(',').omitEmptyStrings();

    public DeployitTester() {
        this.repository = com.xebialabs.deployit.test.repository.InMemoryRepository.REFERENCE.get();
        this.repository.clear();
        this.factory = new RuleBasedPlannerFactory();
        this.planner = factory.plannerWithoutStaging();
        this.inspector = new Inspector();
    }

    /**
     * @param planner
     * @param inspector
     * @param repository
     * @deprecated Use the
     */
    @Deprecated
    public DeployitTester(DeploymentPlanner planner, Inspector inspector, InMemoryRepository repository) {
        this.planner = planner;
        this.inspector = inspector;
        this.repository = repository.wrapped;
    }

    public DeployitTester(DeploymentPlanner planner, Inspector inspector, com.xebialabs.deployit.test.repository.InMemoryRepository repository) {
        this.planner = planner;
        this.inspector = inspector;
        this.repository = repository;
    }

    public void shutdown() {
        if (factory != null) {
            factory.shutdownTheSystem();
        }
    }

    /**
     * @deprecated Just use the default constructor.
     */
    @Deprecated
    public static DeployitTester build() {
        return new DeployitTester();
    }

    public List<Step> resolvePlan(DeltaSpecification spec) {
        return Lists.transform(PlanUtils.flattenPlan(resolve(spec)), StepPlan.stepsWithDeltaToSteps);
    }

    /**
     * @deprecated taskId parameter is no longer necessary.
     */
    @Deprecated
    public List<Step> resolvePlanWithStaging(String taskId, DeltaSpecification spec) {
        return this.resolvePlanWithStaging(spec);
    }

    public List<Step> resolvePlanWithStaging(DeltaSpecification spec) {
        PhasedPlan plan = new DeploymentStager(planner).plan(spec, repository);
        planLogger.info("Generated plan for {}:\n{}\n", firstCallerOnStack("com.xebialabs"), plan.writePlan(new StringWriter()));
        return Lists.transform(PlanUtils.flattenPlan(plan), StepPlan.stepsWithDeltaToSteps);
    }

    public static List<Step> resolvePlan(Plan plan) {
        return Lists.transform(PlanUtils.flattenPlan(plan), StepPlan.stepsWithDeltaToSteps);
    }

    public static List<StepPlan.Checkpoint> resolveCheckpoints(Plan plan) {
        return PlanUtils.resolveCheckpoints(plan);
    }

    public PhasedPlan resolve(DeltaSpecification spec) {
        PhasedPlan plan = planner.plan(spec, repository);
        planLogger.info("Generated plan for {}:\n{}\n", firstCallerOnStack("com.xebialabs"), plan.writePlan(new StringWriter()));
        return plan;
    }

    private static String firstCallerOnStack(String packagePrefix) {
        StackTraceElement[] stackTrace = new Exception().getStackTrace();
        for (int i = stackTrace.length - 1; i >= 0; i--) {
            StackTraceElement frame = stackTrace[i];
            if (frame.getClassName().startsWith(packagePrefix)) {
                return frame.getMethodName();
            }
        }
        return "<unknown>";
    }

    /**
     * @deprecated Use {@link #runInspectionTask(ConfigurationItem)}.
     */
    @Deprecated
    public List<ConfigurationItem> inspect(ConfigurationItem item) {
        com.xebialabs.deployit.test.support.LoggingExecutionContext ctx = new com.xebialabs.deployit.test.support.LoggingExecutionContext(DeployitTester.class);
        try {
            return inspect(item, ctx);
        } finally {
            ctx.destroy();
        }
    }

    /**
     * @deprecated Use {@link #runInspectionTask(ConfigurationItem, TestExecutionContext)}.
     */
    @Deprecated
    public List<ConfigurationItem> inspect(ConfigurationItem item, com.xebialabs.deployit.plugin.api.execution.ExecutionContext ctx) {
        return inspector.inspect(item, ctx);
    }

    public List<ConfigurationItem> runInspectionTask(ConfigurationItem item) {
        TestExecutionContext executionContext = new TestExecutionContext(repository);
        try {
            return runInspectionTask(item, executionContext);
        } finally {
            executionContext.destroy();
        }
    }

    public List<ConfigurationItem> runInspectionTask(ConfigurationItem item, TestExecutionContext executionContext) {
        TestInspectionContext inspectionContext = executionContext.getInspectionContext();
        // Create plan
        Inspector.inspect(item, inspectionContext);

        // Run inspection
        StepExitCode result = executePlan(inspectionContext.getSteps(), executionContext);
        if (result == FAIL) {
            throw new DeployitTesterException("Inspection of " + item + " failed.");
        }

        return new ArrayList<>(inspectionContext.getInspected().values());
    }

    @SuppressWarnings("rawtypes")
    public Deployed generateDeployed(Deployable d, Container c, Type deployedType) {
        return generateDeployed(d, c, deployedType, null);
    }

    @SuppressWarnings("rawtypes")
    public Deployed generateDeployed(Deployable d, Container c, Type deployedType, Map<String, String> placeholders) {
        Descriptor deployed = DescriptorRegistry.getDescriptor(deployedType);
        Deployed<Deployable, Container> configurationItem = deployed.newInstance(c.getId() + "/" + substringAfterLastSlash(d.getId()));
        configurationItem.setDeployable(d);
        configurationItem.setContainer(c);

        Descriptor deployableDescriptor = DescriptorRegistry.getDescriptor(d.getType());
        for (PropertyDescriptor propertyDescriptor : deployed.getPropertyDescriptors()) {
            String name = propertyDescriptor.getName();
            PropertyDescriptor deployablePropertyDescriptor = deployableDescriptor.getPropertyDescriptor(name);
            if (deployablePropertyDescriptor != null) {
                if (propertyDescriptor.getName().equals("placeholders")) {
                    propertyDescriptor.set(configurationItem, newHashMap());
                } else {
                    propertyDescriptor.set(configurationItem, convertCiValuesIfNeeded(deployablePropertyDescriptor.get(d), propertyDescriptor, deployablePropertyDescriptor));
                }
            }
        }

        if (configurationItem instanceof DerivedArtifact<?>) {
            DerivedArtifact<?> da = (DerivedArtifact<?>) configurationItem;
            if (placeholders != null) {
                da.setPlaceholders(placeholders);
            }
            Placeholders.replacePlaceholders(da, new SimpleReplacer());
        }

        return configurationItem;
    }

    private Object convertCiValuesIfNeeded(Object o, PropertyDescriptor deployedPropertyDescriptor, PropertyDescriptor deployablePropertyDescriptor) {
        if (o == null) return null;
        if (deployablePropertyDescriptor.getKind() == STRING && deployedPropertyDescriptor.getKind() == CI) {
            return repository.read((String) o);
        } else if (deployablePropertyDescriptor.getKind() == LIST_OF_STRING && deployedPropertyDescriptor.getKind() == LIST_OF_CI) {
            return newArrayList(Lists.transform((List<String>) o, idToCi));
        } else if (deployablePropertyDescriptor.getKind() == STRING && deployedPropertyDescriptor.getKind() == LIST_OF_CI) {
            return newArrayList(Iterables.transform(STRING_TO_COLLECTION_SPLITTER.split((String) o), idToCi));
        } else if (deployablePropertyDescriptor.getKind() == SET_OF_STRING && deployedPropertyDescriptor.getKind() == SET_OF_CI) {
            return newHashSet(Collections2.transform((Set<String>) o, idToCi));
        } else if (deployablePropertyDescriptor.getKind() == STRING && deployedPropertyDescriptor.getKind() == SET_OF_CI) {
            return newArrayList(Iterables.transform(STRING_TO_COLLECTION_SPLITTER.split((String) o), idToCi));
        } else {
            return o;
        }
    }

    private static String substringAfterLastSlash(String id) {
        int i = id.lastIndexOf('/');
        if (i > -1) {
            return id.substring(i + 1);
        }
        return id;
    }

    public StepExitCode executePlan(List<Step> plan) {
        TestExecutionContext ctx = new TestExecutionContext(getClass(), repository);
        try {
            return executePlan(plan, ctx);
        } finally {
            ctx.destroy();
        }
    }

    public static StepExitCode executePlan(List<Step> steps, ExecutionContext context) {
        StepExitCode result = StepExitCode.PAUSE;
        try {
            // Steps may be added dynamically
            for (int i = 0; i < steps.size(); i++) {
                result = steps.get(i).execute(context);
                if (result == StepExitCode.FAIL) {
                    break;
                }
            }
            return result;
        } catch (Exception e) {
            throw new DeployitTesterException(e);
        }
    }


    public Repository repository() {
        return repository;
    }

    @Deprecated
    public InMemoryRepository getRepository() {
        return new InMemoryRepository(repository);
    }

    @SuppressWarnings("serial")
    public static class DeployitTesterException extends RuntimeException {
        DeployitTesterException(String message, Object... params) {
            super(format(message, params));
        }

        DeployitTesterException(Exception e) {
            super(e);
        }
    }


    private final static Logger planLogger = LoggerFactory.getLogger(StepPlan.class);
}

