package com.xebialabs.deployit.engine.tasker;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.Closeables;
import com.google.common.util.concurrent.Monitor;

import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.execution.StepExecutionStateEvent;
import com.xebialabs.deployit.engine.spi.execution.TaskExecutionStateEvent;

import javassist.util.proxy.ProxyObjectOutputStream;

import static java.lang.String.format;

@SuppressWarnings("serial")
class TaskRecoveryTrigger implements ExecutionStateListener {

    private static final Timer t = new Timer("recovery-writer", true);

    private File recoveryDir;

    private transient AtomicReference<TaskState> taskReference = new AtomicReference<TaskState>();

    private transient Monitor M = new Monitor();

    public TaskRecoveryTrigger(File recoveryDir) {
        this.recoveryDir = recoveryDir;
    }

    @Override
    public void stepStateChanged(StepExecutionStateEvent event) {
        M.enter();
        try {
            startWritingTimer(event.task());
        } finally {
            M.leave();
        }
    }

    @Override
    public void taskStateChanged(TaskExecutionStateEvent event) {
        M.enter();
        try {
            if (!event.currentState().isFinal()) {
                startWritingTimer(event.task());
            } else {
                taskReference.set(null);
                getFile(event.task().getId()).delete();
            }
        } finally {
            M.leave();
        }
    }

    private void startWritingTimer(TaskState task) {
        if (taskReference.compareAndSet(null, task)) {
            t.schedule(new RecoveryTimerTask(), 1000);
            logger.debug("Scheduled Recovery writer for [{}] for 1 second from now", task.getId());
        }
    }

    private File getFile(String id) {
        return new File(recoveryDir, id + ".task");
    }

    private class RecoveryTimerTask extends TimerTask {
        @Override
        public void run() {
            M.enter();
            try {
                // Could have been waiting and it was removed in the meantime.
                if (taskReference.get() != null) {
                    logger.debug("Writing recovery file for: [{}]", taskReference.get());
                    writeRecovery(taskReference.get());
                }
            } catch(Throwable t) {
                logger.error(String.format("Exception writing recovery file for: [{}]", taskReference.get()), t);
            } finally {
                taskReference.set(null);
                M.leave();
            }
        }

        private void writeRecovery(TaskState task) {
            String id = task.getId();
            ObjectOutputStream os = null;
            try {
                os = new ProxyObjectOutputStream(new FileOutputStream(getFile(id)));
                os.writeObject(task);
            } catch (IOException e) {
                logger.error(format("Could not write recovery file for [%s]", id), e);
            } finally {
                Closeables.closeQuietly(os);
            }
        }
    }

    private void readObject(java.io.ObjectInputStream stream) throws IOException, ClassNotFoundException {
        stream.defaultReadObject();
        M = new Monitor();
        taskReference = new AtomicReference<TaskState>();
    }

    private static final Logger logger = LoggerFactory.getLogger(TaskRecoveryTrigger.class);
}
