package com.xebialabs.deployit.engine.replacer;

import java.io.*;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.io.CharStreams;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.local.LocalFile;

import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileReader;
import de.schlichtherle.truezip.file.TFileWriter;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.emptyToNull;
import static java.lang.Math.abs;
import static java.util.regex.Pattern.CASE_INSENSITIVE;
import static java.util.regex.Pattern.COMMENTS;

public class Placeholders {

    public static LoadingCache<String, Pattern> patternMap = CacheBuilder.newBuilder().build(new CacheLoader<String, Pattern>() {
        @Override
        public Pattern load(final String key) throws Exception {
            return Pattern.compile(key, COMMENTS | CASE_INSENSITIVE);
        }
    });

    public static void scanPlaceholders(SourceArtifact artifact, PlaceholderScanner scanner) {
        checkArgument(artifact.getFile() != null, artifact + " has no file");
        checkArgument(artifact.getFile() instanceof LocalFile, "Cannot scan for placeholders in " + artifact + " because its file is a " + artifact.getFile().getClass().getName() + " and not a " + LocalFile.class.getName());

        artifact.setPlaceholders(Sets.<String>newTreeSet());
        if (artifact.hasProperty("scanPlaceholders") && !(Boolean) artifact.getProperty("scanPlaceholders")) {
            return;
        }
        TFile from = asTfile(artifact.getFile());
        try {
            doScanPlaceholders(artifact, from, scanner, true);
        } finally {
            umountQuietly(from);
        }
    }

    private static void doScanPlaceholders(SourceArtifact artifact, TFile from, PlaceholderScanner scanner, boolean isRoot) {
        if (shouldExcludeFile(from, artifact, isRoot)) {
            return;
        }

        if (from.isDirectory()) {
            try {
                for (TFile f : from.listFiles()) {
                    f = asTfile(f);
                    doScanPlaceholders(artifact, f, scanner, false);
                }
            } finally {
                umountQuietly(from);
            }
        } else if (isTextFile(from.getName(), artifact.getTextFileNamesRegex())) {
            artifact.getPlaceholders().addAll(readPlaceholders(from, scanner));
        }
    }

    private static Set<String> readPlaceholders(final TFile from, final PlaceholderScanner scanner) {
        Reader in = null;
        try {
            logger.trace("Reading placeholders from file {}", from.getPath());
            in = new TFileReader(from);
            return scanner.scan(in);
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot scan for placeholders in " + from, exc);
        } catch (RuntimeException exc) {
            throw new RuntimeException("Cannot scan for placeholders in " + from, exc);
        } finally {
            Closeables.closeQuietly(in);
        }
    }

    public static void replacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, PlaceholderReplacer replacer) {
        if (derivedArtifact.getSourceArtifact() == null) {
            derivedArtifact.setFile(null);
        } else {
            OverthereFile fromFile = derivedArtifact.getSourceArtifact().getFile();
            checkArgument(fromFile != null, "%s has no file", derivedArtifact.getSourceArtifact());
            checkArgument(fromFile instanceof LocalFile, "Cannot replace placeholders in %s because its file is not a LocalFile but a %s",
                    derivedArtifact.getSourceArtifact(), fromFile.getClass().getName());

            TFile from = null;
            try {
                from = getTFileWithCorrectDirectoryDetection(fromFile);
                boolean isBinaryFile = from.isFile() && !isTextFile(fromFile.getName(), getTextFileNamesRegex(derivedArtifact));
                if (derivedArtifact.getPlaceholders().isEmpty() || isBinaryFile) {
                    derivedArtifact.setFile(fromFile);
                } else {
                    OverthereFile derivedFile = getOutputFile(derivedArtifact);
                    try {
                        fromFile.copyTo(derivedFile);
                        TFile to = getTFileWithCorrectDirectoryDetection(derivedFile);
                        try {
                            doReplacePlaceholders(derivedArtifact, to, replacer, true);
                            saveArchive(to);
                        } finally {
                            umountQuietly(to);
                        }
                        derivedArtifact.setFile(derivedFile);
                    } finally {
                        checkState(derivedFile.getParentFile().listFiles().size() == 1, "Should only be one file in the deployed dir, was %s", derivedFile.getParentFile().listFiles());
                    }
                }
            } finally {
                umountQuietly(from);
            }
        }
    }

    private static TFile getTFileWithCorrectDirectoryDetection(final OverthereFile derivedFile) {
        final TFile to;// Disable archive detection for directories (useful when the directory is named PetClinic.war)...
        // Also disable archive detection for the parent directory of the artifact (useful when the artifact is named PetClinic.war)
        if (derivedFile.isDirectory()) {
            to = new TFile(derivedFile.getPath(), TArchiveDetector.NULL);
        } else {
            String parentPath = derivedFile.getParentFile().getPath();
            to = new TFile(new TFile(parentPath, TArchiveDetector.NULL), derivedFile.getName(), TArchiveDetector.ALL);
        }
        return to;
    }

    private static void doReplacePlaceholders(DerivedArtifact<? extends SourceArtifact> derivedArtifact, TFile to, PlaceholderReplacer replacer, boolean isRoot) {
        boolean include = !shouldExcludeFile(to, derivedArtifact.getSourceArtifact(), isRoot);

        if (include && to.isDirectory()) {
            try {
                for (TFile t : to.listFiles()) {
                    t = asTfile(t);
                    doReplacePlaceholders(derivedArtifact, t, replacer, false);
                }
            } finally {
                umountQuietly(to);
            }
        } else if (include && isTextFile(to.getName(), getTextFileNamesRegex(derivedArtifact))) {
            replace(to, replacer, derivedArtifact.getPlaceholders());
        } else {
        }
    }

    /**
     * Replacement needs a temporary file to write to.
     */
    private static void replace(final TFile toBeReplaced, final PlaceholderReplacer replacer, Map<String, String> resolution) {
        File tempFile = null;
        try {
            Reader reader = null;
            Writer writer = null;
            tempFile = File.createTempFile(toBeReplaced.getName(), ".tmp");
            try {
                reader = new TFileReader(toBeReplaced);
                writer = new FileWriter(tempFile);
                replacer.replace(reader, writer, resolution);
            } finally {
                Closeables.closeQuietly(reader);
                Closeables.closeQuietly(writer);
            }

            try {
                reader = new FileReader(tempFile);
                writer = new TFileWriter(toBeReplaced);
                CharStreams.copy(reader, writer);
            } finally {
                reader.close();
                writer.close();
            }
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot replace placeholders in " + toBeReplaced, exc);
        } finally {
            if (tempFile != null && !tempFile.delete()) {
                logger.warn("Couldn't delete temporary file: {}", tempFile);
            }
        }
    }

    private static OverthereFile getOutputFile(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
        OverthereFile sourceFile = derivedArtifact.getSourceArtifact().getFile();
        OverthereFile workDir = sourceFile.getParentFile();

        Random r = new Random();
        String baseName = derivedArtifact.getName();
        for (; ; ) {
            String name = baseName + abs(r.nextInt());
            OverthereFile deployedArtifactDir = workDir.getFile(name);

            if (!deployedArtifactDir.exists()) {
                deployedArtifactDir.mkdir();
                return deployedArtifactDir.getFile(sourceFile.getName());
            }
        }
    }

    private static File saveArchive(TFile outputArchive) {
        if (outputArchive.isArchive() && outputArchive.getEnclArchive() == null && outputArchive.isDirectory()) {
            try {
                TFile.umount(outputArchive);
            } catch (IOException exc) {
                throw new RuntimeIOException("Cannot write archive " + outputArchive, exc);
            }
        }

        // Return a regular java.io.File pointing to the file, folder or archive just written
        return new File(outputArchive.getPath());
    }

    private static boolean shouldExcludeFile(final TFile f, final SourceArtifact artifact, boolean isRoot) {

        if(isRoot && artifact instanceof FolderArtifact) {
            return false;
        }

        if (emptyToNull(artifact.getExcludeFileNamesRegex()) == null) {
            return false;
        }

        Pattern excludeFileNamesPattern = patternMap.getUnchecked(artifact.getExcludeFileNamesRegex());
        Matcher excludeFileNamesMatcher = excludeFileNamesPattern.matcher(f.getPath());
        boolean exclude = excludeFileNamesMatcher.matches();
        if (exclude) {
            logger.debug("Excluding file {} from scanning", f);
        }
        return exclude;
    }

    private static boolean isTextFile(final String name, final String textFileNamesRegex) {
        checkNotNull(textFileNamesRegex, "Regex is null");

        Pattern textFileNamesPattern = patternMap.getUnchecked(textFileNamesRegex);
        Matcher textFileNamesMatcher = textFileNamesPattern.matcher(name);
        boolean isTextFile = textFileNamesMatcher.matches();
        logger.debug("Determined {} to be a {} file", name, isTextFile ? "text" : "binary");

        return isTextFile;
    }

    private static String getTextFileNamesRegex(DerivedArtifact<? extends SourceArtifact> derivedArtifact) {
        return derivedArtifact.getSourceArtifact().getTextFileNamesRegex();
    }

    private static TFile asTfile(OverthereFile file) {
        File fromFile = ((LocalFile) file).getFile();
        return asTfile(fromFile);
    }

    private static TFile asTfile(File fromFile) {
        File from = fromFile;
        // Unwrap the TFile so that directory/archive detection works correctly.
        if (from instanceof TFile) {
            from = ((TFile) from).getFile();
        }

        if (from.isDirectory()) {
            return new TFile(from, TArchiveDetector.NULL);
        }
        return new TFile(from);
    }

    private static void umountQuietly(TFile file) {
        if (file != null && file.isArchive() && file.getEnclArchive() == null) {
            try {
                TFile.umount(file);
            } catch (Exception e) {
                logger.error("Couldn't umount [{}], ignoring exception.", file);
                logger.debug("Exception while umounting was: ", e);
            }
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(Placeholders.class);

}
