/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.DataHolder;
import com.thoughtworks.xstream.core.MapBackedDataHolder;
import com.xebialabs.deployit.booter.remote.client.ClientXStreamReaderWriter;
import com.xebialabs.deployit.booter.remote.invoker.DefaultsInvoker;
import com.xebialabs.deployit.booter.remote.invoker.InvocationHandlerInvoker;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitClientException;
import com.xebialabs.deployit.booter.remote.service.StreamingImportingService;
import com.xebialabs.deployit.core.api.ConfigurationService;
import com.xebialabs.deployit.engine.api.*;
import com.xebialabs.deployit.engine.api.task.TaskBlockServiceDefaults;
import com.xebialabs.xltype.serialization.xml.StreamXmlReaderWriter;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;
import org.jboss.resteasy.client.jaxrs.ResteasyClient;
import org.jboss.resteasy.client.jaxrs.ResteasyClientBuilder;
import org.jboss.resteasy.spi.ResteasyProviderFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.BadRequestException;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.ResponseProcessingException;
import jakarta.ws.rs.core.MediaType;
import java.io.*;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.*;
import java.util.stream.Collectors;

/**
 * The {@link com.xebialabs.deployit.booter.remote.Proxies} object is used to access services using RESTEasy.
 * The exposed services must never be cached, because on a re-login in {@link HttpClientHolder} the proxies will be invalidated.
 */
public class Proxies {
    private static final Logger logger = LoggerFactory.getLogger(Proxies.class);
    private final Map<Class<?>, Object> registeredProxies = new HashMap<>();
    private final Map<Class<?>, DefaultsAndMethodLookup> wrapperClassMap = new HashMap<>();
    private final HttpClientHolder httpClientHolder;
    private final BooterConfig booterConfig;
    private final ResteasyClient client;

    public Proxies(HttpClientHolder httpClientHolder, BooterConfig config) {
        this.httpClientHolder = httpClientHolder;
        this.booterConfig = config;
        final XStreamReaderWriter xStreamReaderWriter = config.getXStreamReaderWriter() != null ? config.getXStreamReaderWriter() : new ClientXStreamReaderWriter(config);
        final ResteasyClientBuilder resteasyClientBuilder = (ResteasyClientBuilder) ClientBuilder.newBuilder();
        this.client = (ResteasyClient) resteasyClientBuilder
                .httpEngine(httpClientHolder.createClientHttpEngine())
                .withConfig(ResteasyProviderFactory.getInstance())
                .register(xStreamReaderWriter)
                .register(new StreamXmlReaderWriter())
                .build();
        init(config);
    }

    private void init(BooterConfig config) {
        for (Class<?> clazz : allProxies()) {
            doRegisterProxy(config.getUrl(), clazz, getWrapperClassMap().get(clazz));
        }
    }

    private void doRegisterProxy(String url, Class<?> clazz, DefaultsAndMethodLookup defaultsAndMethodLookup) {
        if (defaultsAndMethodLookup == null) {
            registerProxy(url, clazz);
        } else {
            registerProxy(url, clazz, defaultsAndMethodLookup.getDefaultsClass(), defaultsAndMethodLookup.getLookup());
        }
    }

    public static Logger getLogger() {
        return logger;
    }

    public void registerProxy(String url, Class<?> clazz) {
        logger.debug("Registering Proxy: {}", clazz);
        registeredProxies.put(clazz, wrap(clazz, client.target(url).proxy(clazz)));
    }

    public void registerProxy(String url, Class<?> clazz, Class<?> wrapperClass, MethodHandles.Lookup lookup) {
        logger.debug("Registering Proxy: {} with additional methods of: {}", clazz, wrapperClass);
        wrapperClassMap.put(clazz, new DefaultsAndMethodLookup(wrapperClass, lookup));
        registeredProxies.put(clazz, wrap(wrapperClass, client.target(url).proxy(clazz), lookup));
    }

    private Object wrap(final Class<?> clazz, final Object proxy) {
        InvocationHandler invocationHandler = Proxy.getInvocationHandler(proxy);
        InvocationHandlerInvoker invocationHandlerInvoker = new InvocationHandlerInvoker(invocationHandler);
        return Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class[]{clazz},
                new CliExceptionHandler(invocationHandlerInvoker, booterConfig));
    }

    private Object wrap(final Class<?> clazz, final Object proxy, MethodHandles.Lookup lookup) {
        InvocationHandler invocationHandler = Proxy.getInvocationHandler(proxy);
        DefaultsInvoker defaultsInvoker = new DefaultsInvoker(invocationHandler, lookup);
        return Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(),
                new Class[]{clazz},
                new CliExceptionHandler(defaultsInvoker, booterConfig));
    }

    @SuppressWarnings({"deprecation", "unchecked"})
    private static Iterable<Class<?>> allProxies() {
        ArrayList<Class<?>> classes = new ArrayList<>();
        Collections.addAll(classes, ServerService.class, MetadataService.class, RepositoryService.class,
                ControlService.class, DeploymentService.class, InspectionService.class, PackageService.class,
                ConfigurationService.class, PermissionService.class, RoleService.class, TaskService.class,
                TaskBlockService.class, UserService.class);
        return classes;
    }

    private Map<Class<?>, DefaultsAndMethodLookup> getWrapperClassMap() {
        wrapperClassMap.put(TaskBlockService.class,
                new DefaultsAndMethodLookup(TaskBlockServiceDefaults.class, TaskBlockServiceDefaults.getLookup()));
        return wrapperClassMap;
    }

    @SuppressWarnings("unchecked")
    public <T> T getProxyInstance(Class<T> clazz) {
        return (T) registeredProxies.get(clazz);
    }

    public ServerService getServerService() {
        return getProxyInstance(ServerService.class);
    }

    public MetadataService getMetadataService() {
        return getProxyInstance(MetadataService.class);
    }

    public RepositoryService getRepositoryService() {
        return getProxyInstance(RepositoryService.class);
    }

    public ControlService getControlService() {
        return getProxyInstance(ControlService.class);
    }

    public DeploymentService getDeploymentService() {
        return getProxyInstance(DeploymentService.class);
    }

    public InspectionService getInspectionService() {
        return getProxyInstance(InspectionService.class);
    }

    public PackageService getPackageService() {
        return new StreamingImportingService(httpClientHolder, booterConfig, this);
    }

    public PermissionService getPermissionService() {
        return getProxyInstance(PermissionService.class);
    }

    public RoleService getRoleService() {
        return getProxyInstance(RoleService.class);
    }

    @SuppressWarnings("deprecation")
    public TaskService getTaskService() {
        return getProxyInstance(TaskService.class);
    }

    public TaskBlockService getTaskBlockService() {
        return getProxyInstance(TaskBlockService.class);
    }

    public UserService getUserService() {
        return getProxyInstance(UserService.class);
    }

    public ReportService getReportService() { return getProxyInstance(ReportService.class); }

    public static Proxies reinitialize(Proxies oldProxies, HttpClientHolder httpClientHolder, BooterConfig config) {

        Proxies newProxies = new Proxies(httpClientHolder, config);

        if (oldProxies == null) {
            return newProxies;
        }

        for (Class<?> proxyClass : oldProxies.registeredProxies.keySet()) {
            if (!newProxies.registeredProxies.containsKey(proxyClass)) {
                newProxies.doRegisterProxy(config.getUrl(), proxyClass, oldProxies.getWrapperClassMap().get(proxyClass));
            }
        }


        return newProxies;
    }

    /**
     * Exceptions have special handling in the client.
     *
     * <dl>
     *  <dt>ResponseProcessingException</dt>
     *  <dd>Retrow the wrapped excepion.</dd>
     *  <dt>Bad Request (HTTP 400)</dt>
     *  <dd>
     *   <ul>
     *    <li>If the response has not body, an exception is thrown with some information passed from the server in headers (implemented in {@link com.xebialabs.deployit.booter.remote.resteasy.InternalServerErrorClientResponseInterceptor#tryDeployitException(jakarta.ws.rs.client.ClientResponseContext) Client reponse filter})</li>
     *    <li>If the response has a body, it is unmarshalled and if the object
     *     <ul>
     *      <li>is compatible with the return type of the invoked method, it is returned</li>
     *      <li>is not compatible, a {@link DeployitClientException} is thrown wrapping the object</li>
     *     </ul>
     *    </li>
     *   </ul>
     *  </dd>
     * </dl>
     */
    private static class CliExceptionHandler implements InvocationHandler {
        private final InvocationHandler invocationHandler;
        private final BooterConfig booterConfig;

        public CliExceptionHandler(final InvocationHandler invocationHandler, final BooterConfig booterConfig) {
            this.invocationHandler = invocationHandler;
            this.booterConfig = booterConfig;
        }

        @Override
        public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable {
            try {
                return invocationHandler.invoke(proxy, method, args);
            } catch (ResponseProcessingException rpe) {
                throw rpe.getCause();
            } catch (BadRequestException bre) {
                if (isXMLBody(bre)) {
                    return handleXMLBody(method, bre);
                } else {
                    return handleTextBody(bre);
                }
            }
        }

        private boolean isXMLBody(BadRequestException bre) {
            return bre.getResponse().getHeaders().get("Content-Type").stream()
                                .anyMatch(o -> o.toString().contains(MediaType.APPLICATION_XML));
        }

        private Object handleXMLBody(Method method, BadRequestException bre) {
            logger.debug("Caught a BadRequestException with an entity");
            InputStream inputStream = bre.getResponse().readEntity(InputStream.class);
            XStream xStream = XStreamReaderWriter.getConfiguredXStream();
            final DataHolder dataHolder = new MapBackedDataHolder();
            dataHolder.put("BOOTER_CONFIG", booterConfig.getKey());
            Object o = xStream.unmarshal(XStreamReaderWriter.HIERARCHICAL_STREAM_DRIVER.createReader(
                    inputStream), null, dataHolder);
            if (method.getReturnType().isInstance(o)) {
                return o;
            } else {
                throw new DeployitClientException(o, bre.getResponse().getStatus());
            }
        }

        private Object handleTextBody(BadRequestException bre) {
            logger.debug("Caught a BadRequestException with a plain text body");
            InputStream inputStream = bre.getResponse().readEntity(InputStream.class);
            String response = new BufferedReader(new InputStreamReader(inputStream))
                    .lines().collect(Collectors.joining("\n"));
            throw new DeployitClientException(response, bre.getResponse().getStatus());
        }
    }
}
