/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import com.xebialabs.deployit.booter.remote.resteasy.InternalServerErrorClientResponseInterceptor;
import com.xebialabs.xltype.serialization.rest.LocalDateParamConverter;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;
import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.MethodAnnotationScanner;
import nl.javadude.scannit.scanner.SubTypeScanner;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import org.jboss.resteasy.core.ResteasyDeploymentImpl;
import org.jboss.resteasy.spi.ResteasyDeployment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;


/**
 * The {@link RemoteBooter} can be used to create {@link DeployitCommunicator}'s to connect to different XL Deploy
 * instances. A new {@link DeployitCommunicator} is created by calling {@link #boot(BooterConfig)}.
 * {@link DeployitCommunicator}'s are cached and can be retrieved using the {@link BooterConfig} used to create them or
 * using the configuration key that can be obtained by calling {@link BooterConfig#getKey()}. The {@link RemoteBooter}
 * is thread safe.
 */
public class RemoteBooter {

    private static final ReentrantLock M = new ReentrantLock();
    private static final AtomicBoolean BOOTED = new AtomicBoolean(false);
    private static final Map<String, DeployitCommunicator> COMMUNICATORS = new ConcurrentHashMap<>();
    private static ResteasyDeployment resteasyDeployment;

    public static DeployitCommunicator boot(BooterConfig config) {
        // Double checked locking FTW!
        if (!BOOTED.get()) {
            M.lock();
            try {
                doBoot(config);
            } finally {
                M.unlock();
            }
        }
        return getCommunicator(config);
    }

    private static void doBoot(BooterConfig config) {
        if (!BOOTED.get()) {
            Scannit.boot(Configuration.config()
                    .scan("ai.digital")
                    .scan("com.xebialabs")
                    .scan("ext.deployit") // Deployit Extensions
                    .with(new TypeAnnotationScanner(), new MethodAnnotationScanner(), new SubTypeScanner()));
            bootResteasy();

            DeployitCommunicator communicator = new DeployitCommunicator(config);
            COMMUNICATORS.put(config.getKey(), communicator);
            RemoteDescriptorRegistry.boot(communicator);
            XStreamReaderWriter.registerConfigurationItemAliases();
            BOOTED.set(true);
        }
    }

    public static DeployitCommunicator getCommunicator(BooterConfig config) {
        if (!BOOTED.get()) {
            throw new IllegalStateException("Should first boot the RemoteBooter.");
        }

        if (!COMMUNICATORS.containsKey(config.getKey())) {
            M.lock();
            try {
                if (!COMMUNICATORS.containsKey(config.getKey())) {
                    DeployitCommunicator communicator = new DeployitCommunicator(config);
                    COMMUNICATORS.put(config.getKey(), communicator);
                    RemoteDescriptorRegistry.boot(communicator);
                }
            } finally {
                M.unlock();
            }
        }
        return COMMUNICATORS.get(config.getKey());
    }

    public static DeployitCommunicator getCommunicator(String key) {
        if (!COMMUNICATORS.containsKey(key)) {
            throw new IllegalStateException("The requested DeployitCommunicator has not been initialized.");
        }
        return COMMUNICATORS.get(key);
    }

    public static void shutdown() {
        M.lock();
        try {
            if (BOOTED.get()) {
                for (String configKey : new HashSet<>(COMMUNICATORS.keySet())) {
                    COMMUNICATORS.get(configKey).shutdown();
                }
                resteasyDeployment.stop();
                BOOTED.set(false);
            }
        } finally {
            M.unlock();
        }
    }

    private static void bootResteasy() {
        resteasyDeployment = new ResteasyDeploymentImpl();
        resteasyDeployment.getProviderClasses().add(LocalDateParamConverter.class.getName());
        resteasyDeployment.getProviderClasses().add(InternalServerErrorClientResponseInterceptor.class.getName());
        resteasyDeployment.setAddCharset(true);
        resteasyDeployment.start();
    }

    private static final Logger logger = LoggerFactory.getLogger(RemoteBooter.class);

    static void remove(BooterConfig config) {
        COMMUNICATORS.remove(config.getKey());
    }
}
