/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote;

import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.xebialabs.deployit.plugin.api.reflect.*;

public class RemoteDescriptorRegistry extends BaseDescriptorRegistry {

    private final ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock(true);
    private final Lock readLock = readWriteLock.readLock();
    private final Lock writeLock = readWriteLock.writeLock();

    private Map<Type, Descriptor> descriptors = new HashMap<>();
    private Map<Type, List<Type>> subtypes = new HashMap<>();

    protected RemoteDescriptorRegistry(DescriptorRegistryId id) {
        super(id);
    }

    @Override
    public boolean isLocal() {
        return false;
    }

    @Override
    public boolean isDefault() {
        return false;
    }

    @Override
    public Collection<Descriptor> _getDescriptors() {
        readLock.lock();
        try {
            return descriptors.values();
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public Collection<Type> _getSubtypes(Type supertype) {
        readLock.lock();
        try {
            return getFromTypeMap(subtypes, supertype);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public Descriptor _getDescriptor(Type type) {
        if (!descriptors.containsKey(type)) {
            throw new IllegalArgumentException(String.format("DescriptorRegistry does not know about type [%s]", type));
        }
        readLock.lock();
        try {
            return descriptors.get(type);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    public boolean _exists(Type type) {
        readLock.lock();
        try {
            return descriptors.containsKey(type);
        } finally {
            readLock.unlock();
        }
    }

    public static void boot(DeployitCommunicator communicator) {
        RemoteDescriptorRegistry registry = new RemoteDescriptorRegistry(communicator.getConfig());
        DescriptorRegistry.add(registry);
        registry.reboot(communicator);
    }

    public void reboot(DeployitCommunicator communicator) {
        writeLock.lock();
        try {
            List<Descriptor> list = communicator.getProxies().getMetadataService().listDescriptors();
            reboot(list);
        } finally {
            writeLock.unlock();
        }
    }

    public void reboot(List<Descriptor> list) {
        writeLock.lock();
        try {
            this.descriptors = new HashMap<>();
            this.subtypes = new HashMap<>();

            for (Descriptor descriptor : list) {
                register(descriptor);
                for (Type superType : descriptor.getSuperClasses()) {
                    registerSubtype(superType, descriptor.getType());
                }
                for (Type superType : descriptor.getInterfaces()) {
                    registerSubtype(superType, descriptor.getType());
                }
            }
            // at this point descriptors and subtypes should be considered frozen
            // - i.e. nobody should be able to invoke register and registerSubType
        } finally {
            writeLock.unlock();
        }
    }

    @Override
    public void register(final Descriptor descriptor) {
        descriptors.put(descriptor.getType(), descriptor);
    }

    @Override
    public void registerSubtype(final Type supertype, final Type subtype) {
        addToTypeMap(subtypes, supertype, subtype);
    }

    public Collection<Descriptor> getLoadedDescriptors() {
        return _getDescriptors();
    }

    public Descriptor getLoadedDescriptor(Type type) {
        return _getDescriptor(type);
    }

    public Descriptor getLoadedDescriptor(String prefixedName) {
        return getLoadedDescriptor(lookupType(prefixedName));
    }

    public Descriptor getLoadedDescriptor(String prefix, String name) {
        return getLoadedDescriptor(lookupType(prefix, name));
    }

    private void addToTypeMap(Map<Type, List<Type>> map, Type key, Type value) {
        List<Type> collection = map.get(key);
        if (null == collection) {
            collection = new ArrayList<>();
        }
        collection.add(value);
        map.put(key, collection);
    }

    private Collection<Type> getFromTypeMap(Map<Type, List<Type>> map, Type key) {
        List<Type> list = map.get(key);
        if (null == list) {
            list = new ArrayList<>();
        }
        return list;
    }
}