/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote.resteasy;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientResponseContext;
import javax.ws.rs.client.ClientResponseFilter;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.Provider;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static javax.ws.rs.core.Response.Status.Family.CLIENT_ERROR;
import static javax.ws.rs.core.Response.Status.Family.SERVER_ERROR;

@Provider
public class InternalServerErrorClientResponseInterceptor implements ClientResponseFilter {

    @Override
    public void filter(final ClientRequestContext requestContext, final ClientResponseContext responseContext) throws IOException {
        if (isError(responseContext.getStatusInfo())) {
            logger.info("Handling Error Response, status code: {}", responseContext.getStatus());
            tryDeployitException(responseContext);
            tryUnhandledException(responseContext);
        }
    }

    public boolean isError(final Response.StatusType statusInfo) {
        return statusInfo.getFamily() == SERVER_ERROR || statusInfo.getFamily() == CLIENT_ERROR;
    }

    private void tryUnhandledException(final ClientResponseContext clientResponse) {
        if (clientResponse.getHeaders().containsKey("X-Unhandled-Exception")) {
            logger.debug("Found unhandled header");
            throw new RuntimeException(extractMessage(clientResponse));
        }
    }

    private void tryDeployitException(final ClientResponseContext clientResponse) {
        if (clientResponse.getHeaders().containsKey("X-Deployit-Exception")) {
            logger.debug("Found XL Deploy header");
            if (!(clientResponse.getHeaders().containsKey("X-Entity") && clientResponse.hasEntity())) {
                throw new DeployitClientException(extractMessage(clientResponse), clientResponse.getStatus());
            }
        }
    }

    private String extractMessage(final ClientResponseContext clientResponse) {
        Object exceptionType = clientResponse.getHeaders().getFirst("X-Exception-Type");
        Object path = clientResponse.getHeaders().getFirst("X-Path");
        return String.format("%s [%s]: %s", exceptionType, path, tryToReadEntity(clientResponse));
    }

    private String tryToReadEntity(ClientResponseContext clientResponse) {
        try {
            return IOUtils.toString(clientResponse.getEntityStream(), StandardCharsets.UTF_8);
        } catch (IOException e) {
            return "";
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(InternalServerErrorClientResponseInterceptor.class);
}
