package com.xebialabs.deployit.booter.remote;

import java.io.IOException;
import java.net.MalformedURLException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpRequestInterceptor;
import org.apache.http.client.HttpClient;

import javax.net.ssl.SSLContext;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.AuthCache;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.conn.ssl.BrowserCompatHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.util.EntityUtils;
import org.jboss.resteasy.client.ClientExecutor;
import org.jboss.resteasy.client.core.executors.ApacheHttpClient4Executor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class HttpClientHolder {

    private BooterConfig config;
    private String username;
    private String password;

    private CloseableHttpClient httpClient;
    private ClientExecutor clientExecutor;

    private HttpHost host;
    private HttpHost proxyHost;
    private BasicHttpContext localcontext;

    private Proxies proxies;

    HttpClientHolder(BooterConfig config) {

        this.config = config;
        username = config.getUsername();
        password = config.getPassword();

        if (config.isProxied()) {
            proxyHost = new HttpHost(config.getProxyHost(), config.getProxyPort());
        }

        host = new HttpHost(config.getHost(), config.getPort(), String.format("http%s", config.isSecure() ? "s" : ""));

        initializeClient();

        logger.debug("Client initialized for user={}", username);

        tryAuthenticate(config);
    }

    private void initializeClient() {
        logger.debug("Initializing client for user={}", username);
        PoolingHttpClientConnectionManager connManager = new PoolingHttpClientConnectionManager();
        if (config.getConnectionPoolSize() != 0) {
            connManager.setDefaultMaxPerRoute(config.getConnectionPoolSize());
            connManager.setMaxTotal(config.getConnectionPoolSize());
        }
        HttpClientBuilder clientBuilder = HttpClientBuilder.create();
        clientBuilder.disableCookieManagement();
        clientBuilder.setConnectionManager(connManager);
        for (HttpRequestInterceptor interceptor : config.getHttpRequestInterceptors()) {
            clientBuilder.addInterceptorLast(interceptor);
        }
        localcontext = new BasicHttpContext();

        if (config.isSecure()) {
            setupSecureComms(clientBuilder);
        }

        addCredentials(clientBuilder, username, password);
        enablePreemptiveAuthentication(host);

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(10000)
                .setConnectionRequestTimeout(1000)
                .setSocketTimeout(config.getSocketTimeout()).build();

        // TODO: use JVM configured proxy
        // SystemDefaultRoutePlanner routePlanner = new SystemDefaultRoutePlanner(ProxySelector.getDefault());
        // clientBuilder.setRoutePlanner(routePlanner);

        clientBuilder.setDefaultRequestConfig(requestConfig);

        if (config.isProxied()) {
            logger.debug("Configuring connection via proxy: {}", proxyHost);
            clientBuilder.setProxy(proxyHost);
        }

        httpClient = clientBuilder.build();
        clientExecutor = new ApacheHttpClient4Executor(httpClient, localcontext);

        proxies = Proxies.reinitialize(proxies, this, config);
    }

    private void setupSecureComms(HttpClientBuilder clientBuilder) {
        try {
            // TODO make TrustSelfSignedStrategy configurable
            SSLContext sslContext = SSLContexts.custom().loadTrustMaterial(null, new TrustSelfSignedStrategy()).build();
            // TODO BrowserCompatHostnameVerifier is also a nice one for more configuration options.
            SSLConnectionSocketFactory sslFactory = new SSLConnectionSocketFactory(sslContext, new BrowserCompatHostnameVerifier());
            clientBuilder.setSSLSocketFactory(sslFactory);
        } catch (KeyManagementException | NoSuchAlgorithmException | KeyStoreException e) {
            throw new IllegalStateException("Cannot setup secure communications with XL Deploy", e);
        }
    }

    void shutdown() {
        logger.debug("Shutting down.");
        if (clientExecutor != null) {
            try {
                clientExecutor.close();
            } catch (Exception e) {
                logger.warn("Unable to cleanly shut down clientExecutor.", e);
            }
            clientExecutor = null;
        }
        if (httpClient != null) {
            try {
                httpClient.close();
            } catch (IOException e) {
                logger.warn("Unable to cleanly close httpClient.", e);
            }
            httpClient = null;
        }
        localcontext = null;
    }

    private void addCredentials(HttpClientBuilder clientBuilder, String username, String password) {
        CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(AuthScope.ANY, new UsernamePasswordCredentials(username, password));
        clientBuilder.setDefaultCredentialsProvider(credentialsProvider);
    }

    private void enablePreemptiveAuthentication(HttpHost host) {
        AuthCache authCache = new BasicAuthCache();
        BasicScheme basicAuth = new BasicScheme();
        authCache.put(host, basicAuth);
        localcontext.setAttribute(ClientContext.AUTH_CACHE, authCache);
    }

    public void logout() {
        logger.debug("Logging out.");
        username = null;
        password = null;
    }

    public void loginAs(String username, String password) {
        logger.debug("Attempting to login as {}", username);

        config = BooterConfig.builder(config).withCredentials(username, password).build();
        this.username = config.getUsername();
        this.password = config.getPassword();

        initializeClient();
        logger.debug("Client initialized for user={}", username);
        tryAuthenticate(config);

        RemoteBooter.boot(config);
    }

    public String getUserName() {
        return username;
    }

    private void tryAuthenticate(BooterConfig config) {
        logger.info("Connecting to the XL Deploy server at {}...", config.getUrl());
        try {
            HttpResponse execute = httpClient.execute(new HttpGet(config.getUrl() + "/server/info"), localcontext);
            try {
                final int responseCode = execute.getStatusLine().getStatusCode();
                if (responseCode == 200) {
                    logger.info("Successfully connected.");
                } else if (responseCode == 401 || responseCode == 403) {
                    throw new IllegalStateException("You were not authenticated correctly, did you use the correct credentials?");
                } else if (responseCode == 402) {
                    throw new IllegalStateException(
                            "License not found, invalid, or expired; see the XL Deploy logs. Please contact your XebiaLabs sales representative for a valid license.");
                } else {
                    throw new IllegalStateException("Could contact the server at " + config.getUrl() + " but received an HTTP error code, " + responseCode);
                }
            } finally {
                EntityUtils.consume(execute.getEntity());
                httpClient.getConnectionManager().closeIdleConnections(0, TimeUnit.MILLISECONDS);
            }
        } catch (MalformedURLException mue) {
            throw new IllegalStateException("Could not contact the server at " + config.getUrl(), mue);
        } catch (IOException e) {
            throw new IllegalStateException("Could not contact the server at " + config.getUrl(), e);
        }

    }

    public HttpResponse execute(HttpUriRequest httpRequest) throws ClientProtocolException, IOException {
        return httpClient.execute(httpRequest, localcontext);
    }

    public ClientExecutor createClientExecutor() {
        if (clientExecutor == null) {
            clientExecutor = new ApacheHttpClient4Executor(httpClient, localcontext);
        }

        return clientExecutor;
    }

    public HttpHost getHost() {
        return host;
    }

    public HttpClient getHttpClient() {
        return httpClient;
    }

    public Proxies getProxies() {
        return proxies;
    }

    private static final Logger logger = LoggerFactory.getLogger(HttpClientHolder.class);
}
