/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.booter.remote.xml;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.booter.remote.RemoteDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xltype.serialization.xstream.XStreamProvider;

import static com.xebialabs.xltype.serialization.xstream.Converters.readList;

@XStreamProvider(tagName = "descriptor", readable = Descriptor.class)
public class DescriptorConverter implements Converter {
    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        throw new IllegalStateException("Cannot serialize Descriptors from remote-booter");
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        final TypeProvider typeProvider = new RemoteTypeProvider(context);
        return doUnmarshal(reader, context, typeProvider);
    }

    protected RemoteDescriptor doUnmarshal(final HierarchicalStreamReader reader, final UnmarshallingContext context, final TypeProvider typeProvider) {
        RemoteDescriptor descriptor = new RemoteDescriptor();
        setAttributes(reader, descriptor, typeProvider);
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            if ("description".equals(reader.getNodeName())) {
                descriptor.setDescription(reader.getValue());
            } else if ("icon".equals(reader.getNodeName())) {
                String icon = reader.getValue();
                descriptor.setIcon(icon);
            } else if ("property-descriptors".equals(reader.getNodeName())) {
                List<PropertyDescriptor> list = readList(descriptor, PropertyDescriptor.class, reader, context);
                descriptor.setPropertyDescriptors(list);
            } else if ("control-tasks".equals(reader.getNodeName())) {
                List<MethodDescriptor> list = readList(descriptor, MethodDescriptor.class, reader, context);
                descriptor.setControlTasks(list);
            } else if ("interfaces".equals(reader.getNodeName())) {
                Set<Type> interfaces = new HashSet<>();
                while (reader.hasMoreChildren()) {
                    reader.moveDown();
                    interfaces.add(Type.valueOf(reader.getValue()));
                    reader.moveUp();
                }
                descriptor.setInterfaces(interfaces);
            } else if ("superTypes".equals(reader.getNodeName())) {
                List<Type> superTypes = new ArrayList<>();
                while (reader.hasMoreChildren()) {
                    reader.moveDown();
                    superTypes.add(Type.valueOf(reader.getValue()));
                    reader.moveUp();
                }
                descriptor.setSuperTypes(superTypes);
            }
            reader.moveUp();
        }
        return descriptor;
    }

    private void setAttributes(HierarchicalStreamReader reader, RemoteDescriptor descriptor, TypeProvider typeProvider) {
        descriptor.setType(typeProvider.getType(reader.getAttribute("type")));
        descriptor.setLabel(reader.getAttribute("label"));

        String deployableType = reader.getAttribute("deployableType");
        if (deployableType != null) {
            descriptor.setDeployableType(typeProvider.getType(deployableType));
        }

        String containerType = reader.getAttribute("containerType");
        if (containerType != null) {
            descriptor.setContainerType(typeProvider.getType(containerType));
        }

        String virtual = reader.getAttribute("virtual");
        if ("true".equalsIgnoreCase(virtual)) {
            descriptor.setVirtual();
        }

        String versioned = reader.getAttribute("versioned");
        if ("false".equalsIgnoreCase(versioned)) {
            descriptor.setVersioned(false);
        }

        String inspectable = reader.getAttribute("inspectable");
        if ("true".equalsIgnoreCase(inspectable)) {
            descriptor.setInspectable();
        }

        descriptor.setRootName(readRoot(reader));
    }

    private String readRoot(HierarchicalStreamReader reader) {
        return reader.getAttribute("root");
    }

    @Override
    public boolean canConvert(Class type) {
        return type.equals(Descriptor.class);
    }
}
