package com.xebialabs.deployit.deployment.rules

import java.io.File

import akka.actor._
import com.xebialabs.deployit.deployment.rules.RegistriesActor.Messages._
import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.xlplatform.sugar.ResourcesSugar
import com.xebialabs.xlplatform.watcher.FileWatchActor.Messages._
import grizzled.slf4j.Logging
import nl.javadude.scannit.Scannit

import scala.util.Try
import scalax.file.Path

private[rules] object RegistriesActor {

  def props(fileWatchActor: ActorRef) = Props(new RegistriesActor(fileWatchActor))

  object Messages {

    case class RuleRegistryGet()

    case class StepRegistryGet()

    case class StepMacroRegistryGet()

    case class RuleRegistryReturn(ruleRegistry: RuleRegistry)

    case class StepRegistryReturn(stepRegistry: StepRegistry)

  }

}

private[rules] class RegistriesActor(fileWatchActor: ActorRef) extends Actor with Registries with ResourceScanningSubscription with ActorLogging {

  var steps: StepRegistry = _
  var rules: RuleRegistry = _

  override def preStart() {
    subscribeForFileChanges()
    initStepsAndRules()
  }

  override def receive: Receive = {
    case RuleRegistryGet() => sender() ! RuleRegistryReturn(rules)
    case StepRegistryGet() => sender() ! StepRegistryReturn(steps)
    case Created(_) | Modified(_) | Deleted(_) => initStepsAndRules()
    case WatchStopped(path) => subscribeForFileChange(path)
    case StartWatching(path) => log.debug(s"Watching on $path")
  }

  def initStepsAndRules() = {
    steps = stepRegistry()
    addStepMacros(steps)
    rules = createRuleRegistryWithFallback(steps)
  }

  override def subscribeForFileChange(path: Path) = fileWatchActor ! Watch(path)
}

private[deployit] trait Registries extends Logging {

  def createRuleRegistryWithFallback(steps: StepRegistry) = Try(ruleRegistry(steps)).getOrElse(fallbackRuleRegistry())

  def fallbackRuleRegistry() = {
    val ruleRegistry = new SimpleRuleRegistry()
    ruleRegistry.registerRule(new Rule("TheOnlyFallbackRule", scope = Scope.PRE_PLAN) {
      override def canFire(scopedObject: AnyRef, context: RulePlanningContext) = true

      override def doFire(scopedObject: AnyRef, context: RulePlanningContext): Unit = throw PlannerException(message = "Rules could not be loaded")
    })
    ruleRegistry
  }

  def ruleRegistry(stepRegistry: StepRegistry) = {
    try {
      val ruleRegistry = new SimpleRuleRegistry()
      JavaBasedRuleBuilder.fill(ruleRegistry)
      XmlReader.loadAllAndParse(rulesXmlFileName)(new XmlRuleParser(ruleRegistry, stepRegistry).ruleFromXml)
      XmlReader.loadAllAndParse(rulesXmlFileName)(new XmlRuleParser(ruleRegistry, stepRegistry).disableRulesXml)
      ruleRegistry
    } catch {
      case ex: Exception =>
        error(s"Error while creating rule registry", ex)
        throw ex
    }
  }

  def stepRegistry(scannit: Scannit = Scannit.getInstance()) = {
    val stepRegistry = new SimpleStepRegistry()
    StepClassScanner.fill(stepRegistry, scannit)
    stepRegistry
  }

  def addStepMacros(stepRegistry: StepRegistry) = {
    try {
      XmlReader.loadAllAndParse(rulesXmlFileName)(new XmlStepMacroParser(stepRegistry).stepMacrosFromXml)
      stepRegistry
    }
    catch {
      case ex: Exception =>
        error(s"Error while create step macro registry", ex)
        throw ex
    }
  }

}

private[rules] trait ResourceScanningSubscription extends ResourcesSugar {

  def subscribeForFileChanges() {
    resources(rulesXmlFileName).filter(_.getProtocol == "file").foreach { resource =>
      subscribeForFileChange(Path(new File(resource.toURI)))
    }
  }

  def subscribeForFileChange(path: Path)
}
