package com.xebialabs.deployit.deployment.planner;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.planning.Checkpoint;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.HashMap;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class DefaultDeploymentPlanningContext implements DeploymentPlanningContext {

    private static final Function<DeploymentStep, Step> adapt = new Function<DeploymentStep, Step>() {
        @Override
        public Step apply(DeploymentStep input) {
            return StepAdapter.wrapIfNeeded(input);
        }
    };

    private final HashMap<String, Object> attributes = newHashMap();
    private AppliedDistribution deployedApplication;
    private AppliedDistribution previousDeployedApplication;
    private Repository readOnlyRepository;
    private StepPlan plan;
    private boolean rollback;

    public DefaultDeploymentPlanningContext(AppliedDistribution deployedApplication, AppliedDistribution previousDeployedApplication, Repository readOnlyRepository, StepPlan plan, boolean rollback) {
        this.deployedApplication = deployedApplication;
        this.previousDeployedApplication = previousDeployedApplication;
        this.readOnlyRepository = readOnlyRepository;
        this.plan = plan;
        this.rollback = rollback;
    }

    @Override
    public void addStep(DeploymentStep step) {
        addStep(adapt.apply(step));
    }

    @Override
    public void addSteps(DeploymentStep... steps) {
        addSteps(Iterables.transform(newArrayList(steps), adapt));
    }

    @Override
    public void addSteps(Collection<DeploymentStep> steps) {
        addSteps(Iterables.transform(steps, adapt));
    }

    @Override
    public void addStep(Step step) {
        logger.trace("Adding step [{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        plan.addStep(step);
    }

    @Override
    public void addSteps(Step... steps) {
        logger.trace("Adding steps");
        for (Step step : steps) {
            logger.trace("[{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        }
        plan.addSteps(newArrayList(steps));
    }

    @Override
    public void addSteps(Iterable<Step> steps) {
        logger.trace("Adding steps");
        for (Step step : steps) {
            logger.trace("[{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        }
        plan.addSteps(newArrayList(steps));
    }

    @Override
    public void addCheckpoint(Step step, Checkpoint checkpoint) {
        Delta delta = checkpoint.getDeltaToCheckpoint();
        Operation overrideOperation = checkpoint.getOverrideOperation();
        if (overrideOperation != null) {
            if (delta.getOperation() != Operation.MODIFY) {
                checkArgument(
                    delta.getOperation() == overrideOperation,
                    "The delta's operation (%s) and overrideOperation (%s) should match in non-modify scenarios", delta.getOperation(), overrideOperation);
            } else {
                checkArgument(overrideOperation != Operation.NOOP, "Cannot set a NOOP override operation on a MODIFY delta (%s)", delta);
            }
        }
        plan.getCheckpoints().add(new StepPlan.Checkpoint(delta, step, overrideOperation, checkpoint.getIntermediateCheckpointName()));
    }

    @Override
    public void addCheckpoint(Step step, Delta delta) {
        addCheckpoint(step, new Checkpoint(delta));
    }

    @Override
    public void addCheckpoint(final Step step, final Delta delta, final Operation overrideOperation) {
        addCheckpoint(step, new Checkpoint(delta, null, overrideOperation));
    }

    @Override
    public void addCheckpoint(Step step, Iterable<Delta> deltas) {
        for (Delta delta : deltas) {
            addCheckpoint(step, new Checkpoint(delta));
        }
    }

    @Override
    public void addStepWithCheckpoint(Step step, Checkpoint checkpoint) {
        addStep(step);
        addCheckpoint(step, checkpoint);
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Delta delta) {
        addStepWithCheckpoint(step, new Checkpoint(delta));
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Delta delta, final Operation overrideOperation) {
        addStepWithCheckpoint(step, new Checkpoint(delta, null, overrideOperation));
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Iterable<Delta> deltas) {
        addStep(step);
        for (Delta delta : deltas) {
            addCheckpoint(step, new Checkpoint(delta));
        }
    }

    @Override
    public Object getAttribute(String param) {
        return attributes.get(param);
    }

    @Override
    public void setAttribute(String param, Object value) {
        attributes.put(param, value);
        if (value instanceof ExecutionStateListener) {
            plan.getListeners().add((ExecutionStateListener) value);
        }
    }

    @Override
    public DeployedApplication getDeployedApplication() {
        return (DeployedApplication) deployedApplication;
    }

    @Override
    public DeployedApplication getPreviousDeployedApplication() {
        return (DeployedApplication) previousDeployedApplication;
    }

    @Override
    public AppliedDistribution getAppliedDistribution() {
        return deployedApplication;
    }

    @Override
    public AppliedDistribution getPreviousAppliedDistribution() {
        return previousDeployedApplication;
    }

    @Override
    public Repository getRepository() {
        return readOnlyRepository;
    }

    public boolean isRollback() {
        return rollback;
    }

    private static final Logger logger = LoggerFactory.getLogger(DefaultDeploymentPlanningContext.class);
}
