package com.xebialabs.deployit.deployment.planner;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.rules.StepMetadata;
import com.xebialabs.xlplatform.satellite.Satellite;

import java.io.PrintWriter;
import java.io.Serializable;
import java.io.Writer;
import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.google.common.collect.Sets.newHashSet;
import static java.lang.String.format;
import static java.util.Collections.sort;

public class StepPlan extends ExecutablePlan {

    public static final Function<StepWithPlanningInfo,Step> stepsWithDeltaToSteps = new Function<StepWithPlanningInfo, Step>() {
        public Step apply(StepWithPlanningInfo input) {
            return input.step;
        }
    };

    private ListMultimap<Integer, StepWithPlanningInfo> stepsMap = ArrayListMultimap.create();

    private List<Checkpoint> checkpoints = newArrayList();

    private Delta deltaUnderPlanning;

    private String ruleUnderPlanning;

    public StepPlan(String description, final List<ExecutionStateListener> listeners) {
        super(description, listeners);
    }

    public StepPlan(String description, final List<ExecutionStateListener> listeners, Satellite satellite) {
        super(description, listeners, satellite);
    }

    public StepPlan(String description, Collection<Step> steps, final List<ExecutionStateListener> listeners) {
        super(description, listeners);
        addSteps(steps);
    }

    public StepPlan(String description, Collection<Step> steps, final List<ExecutionStateListener> listeners, Satellite satellite) {
        super(description, listeners, satellite);
        addSteps(steps);
    }

    public StepPlan(String description, Collection<StepWithPlanningInfo> steps, List<Checkpoint> checkpoints, final List<ExecutionStateListener> listeners) {
        super(description, listeners);
        this.checkpoints = checkpoints;

        for (StepWithPlanningInfo step : steps) {
            stepsMap.put(step.getStep().getOrder(), step);
        }
    }

    public StepPlan(String description, Collection<StepWithPlanningInfo> steps, List<Checkpoint> checkpoints, final List<ExecutionStateListener> listeners, Satellite satellite) {
        super(description, listeners, satellite);
        this.checkpoints = checkpoints;

        for (StepWithPlanningInfo step : steps) {
            stepsMap.put(step.getStep().getOrder(), step);
        }
    }

    public void addStep(Step step) {
        stepsMap.put(step.getOrder(), new StepWithPlanningInfo(step, deltaUnderPlanning, ruleUnderPlanning));
    }

    public void addSteps(Collection<Step> steps) {
        for (Step step : steps) {
            addStep(step);
        }
    }

    public List<StepWithPlanningInfo> getStepsWithPlanningInfo() {
        List<Integer> orderedKeys = newArrayList(stepsMap.keySet());
        sort(orderedKeys);

        List<StepWithPlanningInfo> steps = newArrayList();
        for (Integer key : orderedKeys) {
            steps.addAll(stepsMap.get(key));
        }

        return steps;
    }

    public List<Step> getSteps() {
        return transform(getStepsWithPlanningInfo(), stepsWithDeltaToSteps);
    }

    public List<Checkpoint> getCheckpoints() {
        return checkpoints;
    }

    @Override
    public List<Checkpoint> findCheckpoints() {
        return new ArrayList<>(getCheckpoints());
    }

    @Override
    public Writer writePlan(Writer writer) {
        PrintWriter out = new PrintWriter(writer);

        out.println(" * " + this.getDescription());
        out.println("   " + Strings.repeat("-", this.getDescription().length()));

        Multimap<Step, Checkpoint> checkpointsByStep = ArrayListMultimap.create();
        for (Checkpoint checkpoint : checkpoints) {
            checkpointsByStep.put(checkpoint.getStep(), checkpoint);
        }

        int index = 0;
        for (StepWithPlanningInfo swo : getStepsWithPlanningInfo()) {
            index++;
            out.format("%2s. (%3s) - %s  (step: %s, rule: %s)\n", index, swo.getStep().getOrder(), swo.getStep().getDescription(), getStepName(swo.getStep()), swo.getRule());
            if (checkpointsByStep.containsKey(swo.step)) {
                for (Checkpoint checkpoint : checkpointsByStep.get(swo.getStep())) {
                    out.format("            *checkpoint*  %s\n", checkpoint);
                }
            }
        }

        if (index == 0) {
            out.format("<empty plan>");
        }
        return writer;
    }

    private String getStepName(Step step) {
        final StepMetadata stepMetadata = step.getClass().getAnnotation(StepMetadata.class);
        return stepMetadata != null ? stepMetadata.name() : step.getClass().getSimpleName();
    }

    public void setDeltaUnderPlanning(final Delta delta) {
        this.deltaUnderPlanning = delta;
    }

    public void setRuleUnderPlanning(final String rule) {
        this.ruleUnderPlanning = rule;
    }

    public static class StepWithPlanningInfo {
        private Set<Delta> deltas = newHashSet();
        private Step step;
        private String rule;

        public StepWithPlanningInfo(Step step, Delta delta, String rule) {
            this.rule = rule;

            if (delta != null) {
                this.deltas.add(delta);
            }
            this.step = step;
        }

        public StepWithPlanningInfo(Step step, Set<Delta> deltas) {
            this.deltas.addAll(deltas);
            this.step = step;
        }

        public Set<Delta> getDeltas() {
            return deltas;
        }

        public String getRule(){ return rule; }

        public Step getStep() {
            return step;
        }
    }

    public static class Checkpoint implements Serializable {
        private Delta delta;
        private Operation operation;
        private String intermediateCheckpointName;
        private Step step;
        private final String id = UUID.randomUUID().toString();

        public Checkpoint(Delta delta, Step step) {
            this(delta, step, delta.getOperation(), null);
        }

        public Checkpoint(Delta delta, Step step, Operation operation, String intermediateCheckpointName) {
            this.delta = delta;
            this.step = step;
            this.operation = operation != null ? operation : delta.getOperation();
            this.intermediateCheckpointName = intermediateCheckpointName;
        }

        public Delta getDelta() {
            return delta;
        }

        public Step getStep() {
            return step;
        }

        public Operation getOperation() {
            return operation;
        }

        public String getId() {
            return id;
        }

        public String getIntermediateCheckpointName() {
            return intermediateCheckpointName;
        }

        @Override
        public String toString() {
            String prefix = "";
            if (intermediateCheckpointName != null) {
                prefix = format("(intermediate: '%s') ", intermediateCheckpointName);
            }
            switch (operation) {
                case CREATE:
                    return prefix + format("%s %s", operation, delta.getDeployed());
                case MODIFY:
                    return prefix + format("%s from %s to %s", operation, delta.getPrevious(), delta.getDeployed());
                case DESTROY:
                    return prefix + format("%s %s", operation, delta.getPrevious());
                default:
                    return prefix + operation.toString();
            }
        }
    }
}
