package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, Operation}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.xlplatform._
import com.xebialabs.platform.script.jython.{JythonContext, JythonSupport}
import grizzled.slf4j.Logging

abstract class Rule(name: String, scope: Scope, order: Option[Int] = None) extends Logging {
  var enabled: Boolean = true

  def getName = name

  def getScope = scope

  def isEnabled = enabled

  def disable() {
    enabled = false
  }

  def getOrder = order

  /**
   * Check whether this rule allows itself to be fired.
   *
   * @param scopedObject depending on scope: 'deployed' -> the Delta under planning; 'plan' -> the Deltas object containing all Delta objects in this interleaved plan; 'pre-plan' and 'post-plan' -> the DeltaSpecification for the full deployment
   * @param context the rule planning context
   */
  def canFire(scopedObject: AnyRef, context: RulePlanningContext): Boolean

  /**
   * Fires this rule.
   *
   * @param scopedObject same as in {@see canFire(AnyRef, RulePlanningContext)}
   * @param context the rule planning context
   */
  def doFire(scopedObject: AnyRef, context: RulePlanningContext): Unit

  override def toString = s"${this.getClass.getSimpleName}[name: $name, scope: $scope]"
}

trait PlanRule extends ConditionalRule with ContextFactory {
  self: Rule =>

  require(List(Scope.PLAN, Scope.POST_PLAN, Scope.PRE_PLAN).contains(getScope), "Only pre-plan, post-plan and plan are valid scopes for this rule")

  override def canFire(scopedObject: AnyRef, context: RulePlanningContext): Boolean = {
    implicit val pContext = context
    implicit val jContext: JythonContext = jythonContext(scopedObject, getScope)
    evaluateCondition(scopedObject, context)
  }
}

trait DeployedRule extends ConditionalRule with ContextFactory {
  self: Rule =>

  require(getScope == Scope.DEPLOYED, "Only deployed is a valid scope for this rule")

  protected val ciTypes: Iterable[Type]
  protected val operations: Iterable[Operation]

  override def canFire(scopedObject: AnyRef, context: RulePlanningContext): Boolean = {
    implicit val pContext = context
    implicit val jContext  = jythonContext(scopedObject, getScope)
    operationMatch(scopedObject.asInstanceOf[Delta]) && typeMatch(scopedObject.asInstanceOf[Delta]) && evaluateCondition(scopedObject)
  }

  def typeMatch(delta: Delta): Boolean = ciTypes.foldLeft(false)((result, `type`) => result || delta.correctDeployed.getType.instanceOf(`type`))

  def operationMatch(delta: Delta): Boolean = operations.toSeq.contains(delta.getOperation)
}


trait ConditionalRule extends JythonSupport {

  protected val condition: Option[String]

  def evaluateCondition(scopedObject: AnyRef)(implicit jythonContext: JythonContext, planningContext: RulePlanningContext): Boolean =
    condition.fold(true)(evaluateExpression[Boolean](_, expr => s"bool($expr)"))
}