package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.plugin.api.rules.Scope

import scala.collection.mutable
import grizzled.slf4j.Logging

class SimpleRuleRegistry extends RuleRegistry with Logging{

  /**
    * Rule Registry
    */
  private[this] val rulesByScope : SortedMultiMap = new mutable.HashMap[Scope, mutable.Set[Rule]]() with SortedMultiMap

  private[this] val rulesByName = mutable.Map[String, Rule]()

  override def registerRule(rule: Rule) {
    if (isRuleNameRegistered(rule.getName)) {
      throw new IllegalArgumentException("Rule with duplicate name [%s] found." format rule.getName)
    }
    info(s"Registering rule $rule")
    registerRuleByName(rule)
    registerRuleByScope(rule)
  }

  private def isRuleNameRegistered(name: String): Boolean = rulesByName.isDefinedAt(name)

  private def registerRuleByName(rule: Rule) = rulesByName(rule.getName) = rule

  private def registerRuleByScope(rule: Rule) = rulesByScope.addBinding(rule.getScope, rule)

  override def getRule(name: String): Option[Rule] = rulesByName.get(name)

  override def getEnabledRules(scope: Scope) = rulesByScope.getOrElse(scope, mutable.Set[Rule]()).filter(_.isEnabled)

  override def disableRule(name: String) {
    getRule(name) match {
      case Some(rule) =>
        info(s"Disabling rule [$name]")
        rule.disable()
      case _ => warn(s"Rule [$name] cannot be disable since it does not exist in the registry.")
    }
  }

  private[this] trait SortedMultiMap extends mutable.MultiMap[Scope, Rule] {
    override protected def makeSet = mutable.TreeSet[Rule]()(Ordering.by(rule => rule.getOrder.getOrElse(Integer.MAX_VALUE) -> rule.getName))
  }
}
