package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.plugin.satellite._
import com.xebialabs.xlplatform.satellite.Satellite

import scala.collection.convert.wrapAll._

trait Satellites {
  private implicit lazy val actorSystem = SatelliteCommunicatorSystem.actorSystem

  private def pingStepDescription(satellites: Set[Satellite]) = satelliteDescription("Checking connectivity of", satellites)

  private def preparingPhaseDescription(satellites: Set[Satellite]) = satelliteDescription("Preparing task on", satellites)

  private def preparingPlanDescription(satellite: Satellite) = s"Preparing task on satellite ${satellite.getName}"

  private def cleanUpPhaseDescription(satellites: Set[Satellite]) = satelliteDescription("Cleaning up", satellites)

  private[this] def satelliteDescription(verb: String, satellites: Set[Satellite]) = if (satellites.isEmpty) "" else {
    val s = if (satellites.size > 1) "s" else ""
    s"$verb satellite$s ${satelliteNames(satellites)}"
  }

  private def satelliteNames(satellites: Set[Satellite]): String = {
    satellites.map(_.getName).toList.sorted.mkString(", ")
  }

  private def extractSatellitesFromPlans(promotedPlan: PhasedPlan): Set[Satellite] =
    promotedPlan.phases.flatMap(phase => extractSatellites(phase.plan)).toSet

  private[this] def extractSatellites(plan: ExecutablePlan): Set[Satellite] = plan match {
    case executablePlan: ExecutablePlan if executablePlan.satellite != null => Set(executablePlan.satellite)
    case stepPlan: StepPlan => Set.empty
    case composite: CompositePlan => composite.getSubPlans.flatMap(extractSatellites).toSet
  }

  private def addPingSteps(promotedPlan: PhasedPlan): PhasedPlan = {
    promotedPlan.copy(phases = promotedPlan.phases.map(phase => addPingSteps(phase, extractSatellites(phase.plan))).toList)
  }

  private def addPingSteps(phase: PlanPhase, allSatellites: Set[Satellite]): PlanPhase = if (allSatellites.isEmpty) phase else {
    val pingSteps = allSatellites.map(PingSatelliteStep(_))
    val pingStepPlan = new StepPlan(pingStepDescription(allSatellites), pingSteps, phase.getListeners)
    phase.copy(plan = new SerialPlan(
      phase.plan.getDescription,
      List(pingStepPlan, phase.plan),
      phase.getListeners))
  }

  private def addPreparePhase(promotedPlan: PhasedPlan, allSatellites: Set[Satellite]): PhasedPlan = if (allSatellites.isEmpty) promotedPlan else {
    val prepareSatellitePlans = allSatellites.toList.sortBy(_.getName).map { sat =>
      val checkStep = CheckExtensionsStep(sat)
      val sendToSatelliteStep = SendToSatelliteStep(sat)
      new StepPlan(preparingPlanDescription(sat), List(checkStep, sendToSatelliteStep), promotedPlan.getListeners)
    }

    val preparingPlan = if (allSatellites.size == 1) prepareSatellitePlans.head else new ParallelPlan(
      preparingPhaseDescription(allSatellites),
      prepareSatellitePlans,
      promotedPlan.getListeners)

    val prepareSatellitePhase = new PlanPhase(preparingPlan, preparingPhaseDescription(allSatellites), promotedPlan.getListeners)
    promotedPlan.copy(phases = prepareSatellitePhase :: promotedPlan.phases.toList)
  }

  private def addCleanUpSatellitePhase(promotedPlan: PhasedPlan, allSatellites: Set[Satellite]): PhasedPlan = if (allSatellites.isEmpty) promotedPlan else {
    val steps = allSatellites.map(CleanUpSatelliteStep(_))
    val phaseDescription: String = cleanUpPhaseDescription(allSatellites)
    val cleanupPlan = new StepPlan(phaseDescription, steps, promotedPlan.getListeners)
    val cleanUpSatellitePhase = new PlanPhase(cleanupPlan, phaseDescription, promotedPlan.getListeners, true)
    promotedPlan.copy(phases = promotedPlan.phases.toList :+ cleanUpSatellitePhase)
  }

  def prepareForSatelliteExecution(plan: PhasedPlan): PhasedPlan = {
    val satellites = extractSatellitesFromPlans(plan)
    addCleanUpSatellitePhase(addPreparePhase(addPingSteps(plan), satellites), satellites)
  }
}

object Satellites extends Satellites
