package com.xebialabs.deployit.deployment.rules

import java.util

import com.github.drapostolos.typeparser.TypeParser
import com.xebialabs.platform.script.jython.{JythonContext, JythonSupport}

class XmlParameterResolver(stepRegistry: StepRegistry) extends JythonSupport {

  private val parser: TypeParser = TypeParser.newBuilder().build()

  def resolveXmlParameters(stepName: String, parameters: Iterable[XmlParameter])(implicit jythonContext: JythonContext): Map[String, Any] = {
    implicit val stepDescriptor: StepDescriptor = stepRegistry.getStepDescriptor(stepName)
    parameters.map(p => p.name -> resolveXmlParameter(p)).toMap
  }

  private def resolveXmlParameter(parameter: XmlParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext) = parameter match {
    case basic: XmlBasicParameter => resolveBasicParameter(basic)
    case expression: XmlExpressionParameter => resolveExpressionParameter(expression)
    case collection: XmlCollectionParameter => stepDescriptor.parameters.find(_._1 == collection.name) match {
      case Some((name, param)) if param.isFieldOfType[util.Map[_, _]] => resolveMapParameter(collection)
      case Some((name, param)) if param.isFieldOfType[util.List[_]] => resolveListParameter(collection)
      case _ => throw new IllegalArgumentException(s"Cannot resolve parameter ${collection.name} as a collection")
    }
  }

  private def resolveBasicParameter(parameter: XmlBasicParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Any = {
    val valueType = stepDescriptor.parameters.get(parameter.name).map(_.field.getType).getOrElse(classOf[String])
    Option(parameter).map(_.value).filter(_.nonEmpty).map(value => parseValueAsType(value, valueType)).orNull
  }

  private def parseValueAsType(value: String, valueType: Class[_]) = {
    require(parser.isTargetTypeParsable(valueType), s"Value '$value' cannot be parsed as type ${valueType.getName}")
    parser.parse(value, valueType)
  }

  private def resolveExpressionParameter(parameter: XmlExpressionParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext) = {
    evaluateExpression[Any](parameter.expression, ProxyUnwrapper.unwrap)
  }

  private def resolveMapParameter(map: XmlCollectionParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Map[String, Any] = {
    map.parameters.map(p => p.name -> resolveXmlParameter(p)).toMap
  }

  private def resolveListParameter(list: XmlCollectionParameter)(implicit stepDescriptor: StepDescriptor, jythonContext: JythonContext): Seq[Any] = {
    require(list.parameters.map(_.name).toSet.size <= 1, "List parameter can have only value tag specified")
    require(list.parameters.map(_.name).toSet.headOption.getOrElse("value") == "value", "List parameter can have only value tag specified")
    list.parameters.map(resolveXmlParameter).toSeq
  }
}

object ProxyUnwrapper {
  private[rules] val unwrap: JythonSupport.PreprocessExpression = exp => s"_unwrapper.unwrap($exp)"
}