package com.xebialabs.deployit.deployment.stager

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.deployment.stager.DeploymentStager._StagingContext
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.flow._
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.artifact.{SourceArtifact, DerivedArtifact, Artifact}
import grizzled.slf4j.Logging


import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.language.reflectiveCalls

class DeploymentStager(wrappedPlanner: Planner) extends Planner with Logging with PlanSugar {

  type GetStagingTarget = {def getStagingTarget: StagingTarget}

  def plan(spec: DeltaSpecification, repository: Repository): PhasedPlan = {
    val stagingContext = new _StagingContext
    val plan = wrappedPlanner.plan(spec, repository)

    logger.info(s"Staging artifacts for plan: [${plan.getDescription}]")

    val staged = plan.phases.map(stage(_, stagingContext))

    val cleanupPlans = staged.flatMap(_._2).toList

    val newPhases = staged.map(_._1)

    cleanupPlans match {
      case Nil =>
        plan.copy(newPhases)

      case cleanupPlan :: Nil =>
        newPhases += new PlanPhase(cleanupPlan, cleanupPlan.getDescription, cleanupPlan.getListeners, alwaysExecuted = true)
        plan.copy(newPhases)

      case plans =>
        val cleanupPlan = new ParallelPlan(cleanupPlans.head.getDescription, plans, plans.head.getListeners)
        newPhases += new PlanPhase(cleanupPlan, cleanupPlan.getDescription, cleanupPlan.getListeners, alwaysExecuted = true)
        plan.copy(newPhases)

    }
  }

  private def stage(planPhase: PlanPhase, stagingContext: _StagingContext): (PlanPhase, Option[ExecutablePlan]) = {
    val staged = stage(planPhase.plan, stagingContext)
    (planPhase.copy(plan = staged._1), staged._2)
  }

  private def stage(plan: ExecutablePlan, stagingContext: _StagingContext): (ExecutablePlan, Option[ExecutablePlan]) = {
    doStage(plan, stagingContext)
    if (stagingContext.stagingSteps.isEmpty) {
      (plan, None)
    } else {
      def stepsToPlan(steps: Iterable[Step with GetStagingTarget], descriptionPrefix: String): List[StepPlan] = {
        steps.groupBy(_.getStagingTarget).map({ case (t, s) => new StepPlan(s"$descriptionPrefix ${t.getName}", s, plan.getListeners)}).toList
      }

      val stagingPlan = new ParallelPlan("Stage artifacts", stepsToPlan(stagingContext.stagingSteps, "Staging to"), plan.getListeners)
      val cleanupPlan = new ParallelPlan("Clean up staged artifacts", stepsToPlan(stagingContext.cleanupHosts.map(t => new StagedFileCleaningStep(t)), "Clean up staged files on"), plan.getListeners)
      val planWithStaging = new SerialPlan(plan.getDescription, List(stagingPlan, plan), plan.getListeners)
      (planWithStaging, Some(cleanupPlan))
    }
  }

  private def doStage(plan: Plan, stagingContext: _StagingContext) {
    logger.debug(s"Staging for [${plan.getClass.getSimpleName}(${plan.getDescription})]")
    plan match {
      case cp: CompositePlan => cp.getSubPlans.foreach(doStage(_, stagingContext))
      case sp: StepPlan => doStage(sp, stagingContext)
    }
  }

  private def doStage(stepPlan: StepPlan, stagingContext: _StagingContext) {
    stepPlan.getSteps.withFilter(_.isInstanceOf[StageableStep]).foreach(step => doStage(step.asInstanceOf[StageableStep], stagingContext))
  }

  private def doStage(step: StageableStep, stagingContext: _StagingContext) {
    logger.debug(s"Preparing stage of artifacts for step [${step.getDescription}]")
    step.requestStaging(stagingContext)
  }
}

object DeploymentStager extends Logging {
  import java.util.{Map => JMap, HashMap => JHashMap}
  import collection.mutable.{Map => MMap}
  private[stager] class _StagingContext extends StagingContext {
    case class StagingKey(checksum: String, placeholders: JMap[String, String], target: StagingTarget)

    val stagingFiles: MMap[StagingKey, StagingFile] = new mutable.LinkedHashMap[StagingKey, StagingFile]()
    val cleanupHosts: mutable.Set[StagingTarget] = new mutable.HashSet[StagingTarget]()

    def stageArtifact(artifact: Artifact, target: StagingTarget): StagedFile = {
      if (Option(target.getStagingDirectoryPath).forall(_.trim.isEmpty)) {
        new JustInTimeFile(artifact)
      } else {
        val key = artifact match {
          case bda: DerivedArtifact[_] if bda.getSourceArtifact != null => StagingKey(bda.getSourceArtifact.getChecksum, bda.getPlaceholders, target)
          case bda: DerivedArtifact[_] if bda.getSourceArtifact == null => StagingKey(null, bda.getPlaceholders, target)
          case sa: SourceArtifact => StagingKey(sa.getChecksum, new JHashMap(), target)
        }

        if (!stagingFiles.contains(key)) {
          stagingFiles.put(key, new StagingFile(artifact, target))
          cleanupHosts += target
        }

        stagingFiles(key)
      }
    }

    def stagingSteps = stagingFiles.values.map(new StagingStep(_))
  }

}
