package com.xebialabs.deployit.deployment.rules

import java.lang.annotation.Annotation
import java.lang.reflect.Method
import java.util.{List => JList}

import com.xebialabs.deployit.plugin.api.deployment.planning._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Deltas, Operation}
import com.xebialabs.deployit.plugin.api.execution.{Step => DeprecatedStep}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.Scope
import nl.javadude.scannit.Scannit
import org.slf4j.LoggerFactory

import scala.collection.convert.wrapAll._
import scala.collection.immutable.TreeSet

object JavaBasedRuleBuilder {

  val logger = LoggerFactory.getLogger(classOf[JavaBasedRuleBuilder])

  def throwStrategy:List[String] => Unit = errs => {
    if (errs.length > 0) {
      throw new IllegalArgumentException("The following errors were encountered during rule scanning: \n * %s" format errs.mkString("\n * "))
    }
  }

  def logWarningStrategy:List[String] => Unit = errs => {
    if (errs.length > 0) {
      logger.warn("The following errors were encountered during rule scanning: \n * %s" format errs.mkString("\n * "))
    }
  }

  def fill(ruleRegistry:RuleRegistry, scannit:Scannit = Scannit.getInstance, errorStrategy:List[String]=>Unit = throwStrategy) {
    new JavaBasedRuleBuilder(ruleRegistry, scannit).registerAllRules(errorStrategy)
  }
}



class JavaBasedRuleBuilder(ruleRegistry: RuleRegistry, scannit: Scannit) {

  private[rules] def registerAllRules(errorStrategy:List[String]=>Unit) {
    val errMsgs = registerDeployedRules() ++ registerContributors() ++ registerPrePostProcessors()
    errorStrategy(errMsgs)
  }

  private[rules] def registerDeployedRules():List[String] = {
    registerDeployedContributors(classOf[Create], Operation.CREATE) ++
      registerDeployedContributors(classOf[Modify], Operation.MODIFY) ++
      registerDeployedContributors(classOf[Destroy], Operation.DESTROY) ++
      registerDeployedContributors(classOf[Noop], Operation.NOOP)
  }

  private[rules] def registerContributors():List[String] = aggregateErrorMsgs(classOf[Contributor], checkContributors, new JavaContributorInvokerRule(_))

  private[rules] def registerPrePostProcessors():List[String] = {
    aggregateErrorMsgs(classOf[PrePlanProcessor], checkProcessor, m => new JavaProcessorInvokerRule(m, Scope.PRE_PLAN, m.getAnnotation(classOf[PrePlanProcessor]).order())) ++
      aggregateErrorMsgs(classOf[PostPlanProcessor], checkProcessor, m => new JavaProcessorInvokerRule(m, Scope.POST_PLAN, m.getAnnotation(classOf[PostPlanProcessor]).order()))
  }

  private def registerDeployedContributors(annotation: Class[_ <: Annotation], operation: Operation):List[String] = {
    val makeRule: Method => Rule = typeContributor => {
      val t: Type = Type.valueOf(typeContributor.getDeclaringClass)
      new JavaDeployedInvokerRule(t, typeContributor, operation)
    }
    aggregateErrorMsgs(annotation, checkDeployedContributors, makeRule)
  }


  private def checkDeployedContributors(c: Method): List[String] = {
    val parameterTypes: Array[Class[_]] = c.getParameterTypes
    validate(c.getReturnType == classOf[Unit], "DeployedContributor %s should have void return type.", c) ++
      validate(parameterTypes.length <= 2 && parameterTypes.length >= 1, "DeployedContributor %s should take 1 or 2 parameters.", c) ++
      validate(parameterTypes.length > 0 && parameterTypes(0) == classOf[DeploymentPlanningContext], "DeployedContributor %s should take %s as first parameter.", c, classOf[DeploymentPlanningContext]) ++
      (if (parameterTypes.length > 1) {
        validate(parameterTypes(1) == classOf[Delta], "DeployedContributor %s should take %s as second parameter.", c, classOf[Delta])
      } else List[String]())
  }

  private def checkContributors(contributor: Method): List[String] = {
    val parameterTypes: Array[Class[_]] = contributor.getParameterTypes
    validate(contributor.getReturnType == classOf[Unit], "Contributor %s should have void return type.", contributor) ++
      validate(parameterTypes.length == 2, "Contributor %s should take exactly 2 parameters.", contributor) ++
      validate(parameterTypes.length > 0 && parameterTypes(0) == classOf[Deltas], "Contributor %s should take %s as first parameter.", contributor, classOf[Deltas]) ++
      validate(parameterTypes.length > 1 && parameterTypes(1) == classOf[DeploymentPlanningContext], "Contributor %s should take %s as second parameter.", contributor, classOf[DeploymentPlanningContext])
  }

  private def checkProcessor(processor:Method):List[String] = {
    val parameterTypes: Array[Class[_]] = processor.getParameterTypes
    validate(List(classOf[DeprecatedStep[_]], classOf[JList[DeprecatedStep[_]]], classOf[Step], classOf[JList[Step]]).contains(processor.getReturnType),
      "Processor %s should have return type %s or %s.", processor, classOf[Step], classOf[JList[Step]]) ++
      validate(parameterTypes.length == 1, "Processor %s should take a single parameter.", processor) ++
      validate(parameterTypes.length > 0 && parameterTypes(0) == classOf[DeltaSpecification], "Processor %s should take %s as its (first and only) parameter.", processor, classOf[DeltaSpecification])
  }


  private def validate(check:Boolean, msgfmt:String, args:Any*):List[String] = {
    if (!check) List(msgfmt.format( args:_* )) else List()
  }

  private def aggregateErrorMsgs(annotation:Class[_ <: Annotation], validator:Method=>List[String], methodToRuleConverter:Method=>Rule):List[String] = {
    var errs = List[String]()
    implicit val ordering = new ClassHierarchyMethodOrdering
    val methods = TreeSet() ++ scannit.getMethodsAnnotatedWith(annotation).toSet
    methods.foreach(method => {
      val validationErrors = validator(method)
      if (validationErrors.length == 0) { ruleRegistry.registerRule(methodToRuleConverter(method)) }
      errs ++= validationErrors
    })
    errs
  }


  /**
   * - First compare on the Class hierarchy, subclasses before superclasses
   * - Then on the class name
   * - Then on the method name
   */
  class ClassHierarchyMethodOrdering extends Ordering[Method] {
    def compare(method: Method, method1: Method): Int = {
      val class1: Class[_] = method.getDeclaringClass
      val class2: Class[_] = method1.getDeclaringClass
      if (isSuperClass(class1, class2)) {
        -1
      } else if (isSuperClass(class2, class1)) {
        1
      } else {
        val name1: String = class1.getName
        val name2: String = class2.getName
        val nameComparison: Int = name1.compareTo(name2)
        if (nameComparison == 0) {
          method.getName.compareTo(method1.getName)
        } else {
          nameComparison
        }
      }
    }

    private[rules] def isSuperClass(c1: Class[_], c2: Class[_]): Boolean = {
      c1.isAssignableFrom(c2) && !(c1.getName == c2.getName)
    }
  }
}
