package com.xebialabs.deployit.deployment.orchestrator;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.*;

import com.xebialabs.deployit.engine.spi.orchestration.Orchestration;
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrator;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.CompositePackage;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeploymentPackage;
import com.xebialabs.deployit.plugin.api.udm.Version;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.xebialabs.deployit.engine.spi.orchestration.Orchestrations.interleaved;
import static com.xebialabs.deployit.engine.spi.orchestration.Orchestrations.serial;

@Orchestrator.Metadata(name = "composite-package")
public class CompositePackageOrchestrator implements Orchestrator {
    @Override
    public Orchestration orchestrate(DeltaSpecification specification) {
        Version version = specification.getDeployedApplication().getVersion();
        if (version instanceof CompositePackage) {
            CompositePackage p = (CompositePackage) version;

            final List<String> ids = newArrayList(transform(getAllDeploymentPackages(p), new Function<Version, String>() {
                @Override
                public String apply(Version input) {
                    return input.getId();
                }
            }));

            if (specification.getOperation() == Operation.DESTROY) {
                Collections.reverse(ids);
            }

            final ImmutableListMultimap<String, Delta> index = Multimaps.index(specification.getDeltas(), new Function<Delta, String>() {
                public String apply(Delta input) {
                    final String id = getDeployed(input).getDeployable().getId();
                    Collection<String> filter = Collections2.filter(ids, new Predicate<String>() {
                        @Override
                        public boolean apply(String input) {
                            return id.startsWith(input);
                        }
                    });
                    if (filter.isEmpty() || filter.size() > 1) {
                        throw new IllegalStateException(String.format("Found no or more than 1 DeploymentPackage (%s) which could have been the source of [%s]", filter, id));
                    }
                    return Iterables.get(filter, 0);
                }
            });

            return serial(transform(ids, new Function<String, Orchestration>() {
                @Override
                public Orchestration apply(String input) {
                    return interleaved(index.get(input));
                }
            }));

        } else {
            return new DefaultOrchestrator().orchestrate(specification);
        }
    }

    private Deployed<?, ?> getDeployed(Delta input) {
        return input.getDeployed() != null ? input.getDeployed() : input.getPrevious();
    }

    private List<Version> getAllDeploymentPackages(CompositePackage comp) {
        return FluentIterable.from(comp.getPackages()).transformAndConcat(new Function<Version, List<Version>>() {
            @Override
            public List<Version> apply(Version version) {
                List<Version> results = new ArrayList<Version>();
                if (version instanceof CompositePackage) {
                    return getAllDeploymentPackages((CompositePackage) version);
                } else if (version instanceof DeploymentPackage) {
                    results.add(version);
                }
                return results;
            }
        }).toImmutableList();
    }

}
