package com.xebialabs.deployit.inspection;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.InspectionExecutionContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionStep;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;

public class InspectionContext implements InspectionExecutionContext, InspectionPlanningContext, DeploymentExecutionContext {

    private ExecutionContext context;

    private Map<String, Object> attributes;
    private Logger stepLogger;

    private List<InspectionStep> steps = newArrayList();
    private AtomicInteger currentStepIndex = new AtomicInteger(0);
    private Map<String, ConfigurationItem> discovered = newHashMap();
    private Map<String, ConfigurationItem> inspected = newHashMap();

    private List<String> capturedError = newArrayList();


    public InspectionContext(final ExecutionContext context) {
        this.context = checkNotNull(context);
    }

    public InspectionContext(final Map<String, Object> attributes) {
        this.attributes = checkNotNull(attributes);
    }

    public void startStepExecution(InspectionStep step) {
        if (attributes != null) {
            this.stepLogger = LoggerFactory.getLogger(step.getClass());
        }
    }

    public void startStepExecution(Step step) {
        if (attributes != null) {
            this.stepLogger = LoggerFactory.getLogger(step.getClass());
        }
    }

    public void addStep(InspectionStep step) {
        this.steps.add(step);
    }

    @Override
    public void discovered(ConfigurationItem item) {
        this.discovered.put(item.getId(), item);
    }

    @Override
    public Map<String, ConfigurationItem> getDiscovered() {
        return discovered;
    }

    @Override
    public void inspected(ConfigurationItem item) {
        if (discovered.remove(item.getId()) == null) {
            logger.warn(item + " has been inspected without being discovered");
        }
        this.inspected.put(item.getId(), item);
    }

    @Override
    public Map<String, ConfigurationItem> getInspected() {
        return inspected;
    }

    @Override
    public void logOutput(String output) {
        if (context != null) {
            context.logOutput(output);
        }
        if (stepLogger != null) {
            stepLogger.info(output);
        }
    }

    @Override
    public void logError(String error) {
        if (context != null) {
            context.logError(error);
        }
        if (stepLogger != null) {
            stepLogger.error(error);
        }
        this.capturedError.add(error);
    }

    @Override
    public void logError(String error, Throwable t) {
        if (context != null) {
            context.logError(error, t);
        }
        if (stepLogger != null) {
            stepLogger.error(error, t);
        }
        this.capturedError.add(error);
        if (t != null) {
            StringWriter sw = new StringWriter();
            t.printStackTrace(new PrintWriter(sw));
            this.capturedError.add(sw.toString());
        }
    }

    @Override
    public Object getAttribute(String name) {
        if (context != null) {
            return context.getAttribute(name);
        } else if (attributes != null) {
            return attributes.get(name);
        } else {
            throw new IllegalStateException(this + " has no context and no attributes");
        }
    }

    @Override
    public void setAttribute(String name, Object object) {
        if (context != null) {
            context.setAttribute(name, object);
        }
        if (attributes != null) {
            attributes.put(name, object);
        }
    }

    public List<InspectionStep> getSteps() {
        return steps;
    }

    public InspectionStep getNextStep() {
        if (currentStepIndex.get() < steps.size()) {
            return steps.get(currentStepIndex.getAndIncrement());
        }
        return null;
    }

    public List<String> getCapturedError() {
        return capturedError;
    }

    private static Logger logger = LoggerFactory.getLogger(InspectionContext.class);

}
