package com.xebialabs.deployit.engine.tasker

import com.xebialabs.deployit.engine.api.execution._
import java.util
import org.joda.time.DateTime
import collection.convert.wrapAll._
import akka.actor.ActorSystem
import grizzled.slf4j.Logger
import scala.collection.mutable
import com.xebialabs.deployit.engine.tasker.step.PauseStep
import java.io._
import javassist.util.proxy.ProxyObjectInputStream
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.StepStateEvent
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent
import com.google.common.io.Closeables
import java.util.UUID
import com.xebialabs.deployit.engine.api.execution.{StepExecutionState => SES}
import com.xebialabs.deployit.engine.api.execution.{TaskExecutionState2 => TES}
import scala.util.{Failure, Success, Try}

object Task {
  val logger = Logger.apply(classOf[Task])

  def apply(file: File): Option[Task] = {
    logger.info(s"Recovering task [$file]")
    var is: ObjectInputStream = null
    try {
      is = new ProxyObjectInputStream(new FileInputStream(file))
      val t: Task = is.readObject.asInstanceOf[Task]
      t.recovered()
      Some(t)
    } catch {
      case e: ClassNotFoundException =>
        logger.error(s"Could not find serialized class in recovery file [$file]", e)
        None
      case e: IOException =>
        logger.error(s"Could not read recovery file [$file]", e)
        None
      case e: RuntimeException =>
        logger.error(s"Could not read recovery file [$file]", e)
        None
    } finally {
      Closeables.closeQuietly(is)
    }
  }
}

// Wondering whether task shouldn't be an actor itself?
class Task(id: TaskId, spec: TaskSpecification) extends TaskWithSteps with Serializable {
  import Task.logger
  def this(id: TaskId, block: Block) = this(id, new TaskSpecification("", "", block))

  var startDate: DateTime = null
  var completionDate: DateTime = null
  var scheduledDate: DateTime = null
  var owner = spec.getOwner
  var failureCount = 0
  var state = TES.UNREGISTERED
  var block: Block = spec.getBlock
  def steps: IndexedSeq[((StepPath, TaskStep), Int)] = fetchStepSeq
  val context = new NestedTaskExecutionContext(None, spec)

  if (spec.getInspectionContext != null) {
    spec.getInspectionContext.registerTask(this)
  }

  private[this] def fetchStepSeq: IndexedSeq[((StepPath, TaskStep), Int)] = block.getSteps().zipWithIndex.toIndexedSeq

  def getSpecification = spec

  def getSteps: util.List[StepState] = steps.map(_._1._2).toList

  def getStep(nr: Int): StepState = steps(stepNrToIndex(nr))._1._2

  def getState: TaskExecutionState = state.toTaskExecutionState

  def getId: String = id

  def getDescription: String = spec.getDescription

  def getStartDate: DateTime = startDate

  def getCompletionDate: DateTime = completionDate

  def getNrSteps: Int = steps.size

  def getCurrentStepNr: Int = getCurrentStepNrs.headOption.map(Integer2int).getOrElse(if (state == TES.EXECUTED) steps.size else 0)

  def getCurrentStepNrs: util.List[Integer] = {

    val mySteps = steps

    def stepByIdxIs(idx: Int, s: SES) =
      mySteps.find(_._2 == idx).map(_._1._2.getState) == Some(s)

    val res: util.List[Integer] = mySteps.collect({
      case ((_, step), idx) if Set(SES.EXECUTING, SES.FAILED, SES.PAUSED).contains(step.getState) => int2Integer(idx + 1)
    }).toList match {
      case Nil if state == TES.EXECUTED => List(int2Integer(mySteps.size))
      case Nil if Set(TES.STOPPED, TES.ABORTED).contains(state) => mySteps.map(_._2).collect {
        case idx if stepByIdxIs(idx, SES.DONE) && stepByIdxIs(idx + 1, SES.PENDING) => int2Integer(idx + 1)
      }
      case Nil => List(int2Integer(0))
      case x@_ => x
    }

    res
  }

  def getMetadata: util.Map[String, String] = spec.getMetadata

  def getFailureCount: Int = failureCount

  def getOwner: String = owner

  def setOwner(newOwner: String) {
    this.owner = newOwner
  }

  def getState2: TaskExecutionState2 = state

  def getTempWorkDir: File = spec.getTempWorkDir

  private[tasker] def setState(state: TaskExecutionState2) {
    this.state = state
  }

  private[tasker] def recordFailure {
    failureCount += 1
  }

  private[tasker] def recordStart {
    if (startDate == null) {
      startDate = new DateTime
    }
  }

  private[tasker] def recordCompletion {
    completionDate = new DateTime
  }

  private[tasker] def getContext: NestedTaskExecutionContext = context

  private[tasker] def getBlock = block

  private[this] def setStateAndNotify(state: StepExecutionState, step: TaskStep)(implicit system: ActorSystem) {
    val oldState = step.getState
    step.setState(state)
    system.eventStream.publish(StepStateEvent(getId, UUID.randomUUID().toString, this, step, oldState, state, None))
  }

  private[tasker] def setTaskStateAndNotify(newState: TaskExecutionState2)(implicit system: ActorSystem) {
    val oldState = state
    state = newState
    logger.info(s"Publishing state change $oldState -> $newState")
    system.eventStream.publish(TaskStateEvent(getId, this, oldState, newState))
  }

  val skippableTaskStates = Set(TES.PENDING, TES.STOPPED, TES.FAILED, TES.ABORTED)

  private[tasker] def skip(stepNrs: List[Int])(implicit system: ActorSystem) {
    if (!skippableTaskStates.contains(state)) throw new TaskerException(s"Task [$id] should be PENDING or STOPPED, but was [$state]")
    Try(stepNrs.foreach {
      stepNr =>
        val step: ((StepPath, TaskStep), Int) = steps(stepNr - 1)
        logger.info(s"Trying to skip step ${step._1}")
        if (!step._1._2.canSkip) throw new IllegalArgumentException(s"Step [${step._1}] cannot be skipped")
        setStateAndNotify(StepExecutionState.SKIP, step._1._2)
    }) match {
      case Success(u) =>
      case Failure(ex) => throw new TaskerException(ex, "Could not skip all steps")
    }
  }

  private[tasker] def unskip(stepNrs: List[Int])(implicit system: ActorSystem) {
    if (!skippableTaskStates.contains(state)) throw new TaskerException(s"Task [$id] should be PENDING or STOPPED, but was [$state]")
    Try(stepNrs.foreach {
      stepNr =>
        val step: ((StepPath, TaskStep), Int) = steps(stepNr - 1)
        logger.info(s"Trying to unskip step ${step._1}")
        if (!step._1._2.isMarkedForSkip) throw new IllegalArgumentException(s"Step [${step._1}] cannot be skipped")
        setStateAndNotify(StepExecutionState.PENDING, step._1._2)
    }) match {
      case Success(u) =>
      case Failure(ex) => throw new TaskerException(ex, "Could not skip all steps")
    }
  }

  private[tasker] def moveStep(stepNr: Int, newPosition: Int) {
    if (state != TES.PENDING || getCurrentStepNr != 0) throw new TaskerException(s"Task [$id] should not have run when moving steps")
    if (!block.isInstanceOf[StepBlock]) throw new TaskerException("Cannot move steps when the task consists of multiple blocks")
    val oldIdx: Int = stepNrToIndex(stepNr)
    val newIdx: Int = stepNrToIndex(newPosition)
    
    val stepBlock: StepBlock = block.asInstanceOf[StepBlock]
    val stepBuffer: mutable.Buffer[TaskStep] = stepBlock.steps.toBuffer
    val removed: TaskStep = stepBuffer.remove(oldIdx)
    stepBuffer.insert(newIdx, removed)

    block = stepBlock.copy(steps = stepBuffer.toList)
//    steps = fetchStepSeq
  }

  def addPause(position: Int) {
    if (!canBeQueued) throw new TaskerException(s"Task [$id] should be PENDING, STOPPED, ABORTED or QUEUED but was [$state]")
    if (!block.isInstanceOf[StepBlock]) throw new TaskerException("Cannot add pause step when the task consists of multiple blocks")
    if (getCurrentStepNr >= position) throw new IllegalArgumentException(s"Can only add pause steps after the current execution position (currently: [$getCurrentStepNr], requested: [$position])")
    val stepBlock: StepBlock = block.asInstanceOf[StepBlock]
    val stepBuffer = stepBlock.steps.toBuffer
    stepBuffer.insert(stepNrToIndex(position), new TaskStep(new PauseStep))

    block = stepBlock.copy(steps = stepBuffer.toList)
//    steps = fetchStepSeq
  }

  def recovered() {
    state = state match {
      case TES.EXECUTING | TES.STOPPING | TES.FAILING | TES.ABORTING => TES.FAILED
      case _ => state
    }

    block = block.recovered()
  }

  private[tasker] def stepNrToIndex(stepNr: Int): Int = {
    if (stepNr <= 0 || stepNr > steps.size) throw new IllegalArgumentException(s"Not a valid step number [$stepNr]")
    stepNr - 1
  }

  def canBeQueued = util.EnumSet.of(TES.PENDING, TES.STOPPED, TES.ABORTED, TES.FAILED).contains(state)

  override def toString: String = s"Task[$id, $state]"

  def getScheduledDate: DateTime = scheduledDate

  def setScheduledDate(scheduledDate: DateTime) {
    this.scheduledDate = scheduledDate
  }
}
