package com.xebialabs.deployit.engine.tasker

import scala.collection.mutable.{ListBuffer, ArrayBuffer}
import java.util

sealed trait Block {
  def id: BlockId

  def state: BlockState

  def getStepList(): java.util.List[TaskStep]

  def getSteps(path: StepPath = StepPath(id)): Seq[(StepPath, TaskStep)] = this match {
    case x: CompositeBlock => x.blocks.flatMap(b => b.getSteps(path / b))
    case x: StepBlock => x.steps.zipWithIndex.map({case (s, i) => (path / (i + 1), s)})
    case x: ExpandableStepBlock => x.steps.zipWithIndex.map({case (s, i) => (path / (i + 1), s)})
  }


  def recovered(): Block = {
    import BlockState._
    if (Set(PENDING, EXECUTED, FAILED, STOPPED, ABORTED).contains(state)) this
    else {
      val recoveredState: BlockState = state match {
        case EXECUTING | STOPPING | FAILING => FAILED
        case ABORTING => ABORTED
        case _ => state
      }
      this match {
        case x: ParallelBlock => x.copy(state = recoveredState, blocks = x.blocks.map(_.recovered().asInstanceOf[ImmutableBlock]))
        case x: SerialBlock => x.copy(state = recoveredState, blocks = x.blocks.map(_.recovered().asInstanceOf[ImmutableBlock]))
        case x: StepBlock => x.copy(state = recoveredState, steps = x.steps.map(_.recovered()))
        case x: ExpandableStepBlock => x.copy(state = recoveredState, steps = x.steps.map(_.recovered()))
      }
    }
  }

  // Butt UGLY!
  def newState[A <: Block](state: BlockState): A = this match {
    case x: ParallelBlock => x.copy(state = state).asInstanceOf[A]
    case x: SerialBlock => x.copy(state = state).asInstanceOf[A]
    case x: StepBlock => x.copy(state = state).asInstanceOf[A]
    case x: ExpandableStepBlock => x.copy(state = state).asInstanceOf[A]
  }
}

sealed trait ImmutableBlock extends Block {

  import scala.collection.JavaConversions._
  def getStepList(): java.util.List[TaskStep] = getSteps(StepPath("/")).map(_._2)
}


sealed trait CompositeBlock extends ImmutableBlock {
  def blocks: Seq[ImmutableBlock]

  private[this] def replaceBlock[A <: ImmutableBlock](f: Seq[ImmutableBlock] => Seq[ImmutableBlock]): A = this match {
    case x: ParallelBlock => x.copy(blocks = f(x.blocks)).asInstanceOf[A]
    case x: SerialBlock => x.copy(blocks = f(x.blocks)).asInstanceOf[A]
  }

  def replaceBlock[A <: ImmutableBlock](block: ImmutableBlock): A = replaceBlock((blocks: Seq[ImmutableBlock]) => blocks.map(b => if (b.id == block.id) block else b))

  def determineNewState(block: ImmutableBlock, remainingBlocks: Seq[ImmutableBlock]): BlockState
}

sealed trait StepContainingBlock[A <: Seq[TaskStep]] extends Block {
  def steps: A
}

case class ParallelBlock(id: BlockId, state: BlockState, blocks: Seq[ImmutableBlock]) extends CompositeBlock {
  def determineNewState(block: ImmutableBlock, remainingBlocks: Seq[ImmutableBlock]): BlockState = {
    import BlockState._
    (state, block.state) match {
      case (PENDING | EXECUTED | FAILED | STOPPED | ABORTED, _)   => throw new IllegalStateException(s"Cannot receive state updates when $state")
      case (ABORTING, EXECUTED | FAILED | STOPPED) | (_, ABORTED) => if (remainingBlocks.isEmpty) ABORTED else ABORTING
      case (ABORTING, _) | (_, ABORTING)                          => ABORTING
      case (FAILING, STOPPED | EXECUTED) | (_, FAILED)            => if (remainingBlocks.isEmpty) FAILED else FAILING
      case (FAILING, _) | (_, FAILING)                            => FAILING
      case (STOPPING, EXECUTED) | (_, STOPPED)                    => if (remainingBlocks.isEmpty) STOPPED else STOPPING
      case (_, STOPPING)                                          => block.state
      case (EXECUTING, EXECUTED)                                  => if (remainingBlocks.isEmpty) EXECUTED else EXECUTING
      case (_, EXECUTING)                                         => state
    }
  }
}
case class SerialBlock(id: BlockId, state: BlockState, blocks: Seq[ImmutableBlock]) extends CompositeBlock {
  def determineNewState(block: ImmutableBlock, remainingBlocks: Seq[ImmutableBlock]): BlockState = {
    import BlockState._
    (state, block.state) match {
      case (_, EXECUTED) => if (remainingBlocks.isEmpty) EXECUTED else state
      case _ => block.state
    }
  }

}
case class StepBlock(id: BlockId, state: BlockState, steps: Seq[TaskStep]) extends StepContainingBlock[Seq[TaskStep]] with ImmutableBlock
case class ExpandableStepBlock(id: BlockId, state: BlockState, steps: ListBuffer[TaskStep]) extends StepContainingBlock[ListBuffer[TaskStep]] {
  import collection.JavaConversions._
  def getStepList(): util.List[TaskStep] = steps
}

case class StepPath(path: String) {
  def /(b: Block) = StepPath(s"$path/${b.id}")
  def /(i: Int) = StepPath(s"$path/$i")

  def asStepNr = path.split('/').last.toInt
}

object Blocks {
  import collection.convert.wrapAll._

  def parallel(id: BlockId, blocks: ImmutableBlock*) = ParallelBlock(id, BlockState.PENDING, blocks)
  // To be used from Java
  def parallel(id: BlockId, blocks: util.List[ImmutableBlock]) = ParallelBlock(id, BlockState.PENDING, blocks)

  def serial(id: BlockId, blocks: ImmutableBlock*) = SerialBlock(id, BlockState.PENDING, blocks)
  // To be used from Java
  def serial(id: BlockId, blocks: util.List[ImmutableBlock]) = SerialBlock(id, BlockState.PENDING, blocks)

  def steps(id: BlockId, steps: List[TaskStep]) = StepBlock(id, BlockState.PENDING, steps)
  // To be used from Java
  def steps(id: BlockId, steps: util.List[TaskStep]) = StepBlock(id, BlockState.PENDING, steps)

  def expandableSteps(id: BlockId, steps: ListBuffer[TaskStep]) = ExpandableStepBlock(id, BlockState.PENDING, steps)

  // To be used from Java
  def expandableSteps(id: BlockId, steps: util.List[TaskStep]) = ExpandableStepBlock(id, BlockState.PENDING, ListBuffer(steps:_*))
}
