/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.placeholders

import com.xebialabs.deployit.plugin.api.udm.artifact.{DerivedArtifact, FolderArtifact, SourceArtifact, TranscodableSourceArtifact}
import com.xebialabs.overthere.{ConnectionOptions, OverthereFile}
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.io.ArtifactIOUtils._
import com.xebialabs.xldeploy.packager.io.SupportedArchiveExtensions.archiveTypeMatchesExtension
import com.xebialabs.xldeploy.packager.io._
import com.xebialabs.xldeploy.packager.placeholders.DerivedArtifactEnricher._
import com.xebialabs.xldeploy.packager.placeholders.PlaceholdersUtil._
import com.xebialabs.xldeploy.packager.transcode.{Ebcdic, Skip, TranscodeSpec}
import grizzled.slf4j.Logger
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveOutputStream}
import org.apache.commons.compress.archivers.jar.JarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry
import org.apache.commons.io.IOUtils
import org.apache.commons.lang3.SystemUtils
import org.slf4j.LoggerFactory

import java.io._
import java.nio.charset.Charset.defaultCharset
import java.nio.file.Files
import java.nio.file.attribute.PosixFileAttributeView
import java.util.UUID
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Using}

object DerivedArtifactEnricher {

  private[DerivedArtifactEnricher] val logger = new Logger(LoggerFactory.getLogger(classOf[DerivedArtifactEnricher]))

  def apply(da: DerivedArtifact[_ <: SourceArtifact], streamerFactory: StreamerFactory): DerivedArtifactEnricher = {
    implicit val transcodeSpec: TranscodeSpec = new TranscodeSpec(new ConnectionOptions(), da.getSourceArtifact.asInstanceOf[TranscodableSourceArtifact])
    new DerivedArtifactEnricher(da)(streamerFactory, transcodeSpec)
  }

  def apply(da: DerivedArtifact[_ <: SourceArtifact],streamerFactory: StreamerFactory,dest: OverthereFile): DerivedArtifactEnricher = {
    implicit val transcodeSpec: TranscodeSpec = new TranscodeSpec(dest.getConnection.getOptions, da.getSourceArtifact.asInstanceOf[TranscodableSourceArtifact])
    new DerivedArtifactEnricher(da)(streamerFactory, transcodeSpec)
  }

  private[DerivedArtifactEnricher] def shouldReplaceArtifactPlaceholders(da: DerivedArtifact[_ <: SourceArtifact]): Boolean =
    da.getSourceArtifact.shouldScanPlaceholders() && da.getPlaceholders != null && da.getPlaceholders.size() > 0

}

class DerivedArtifactEnricher(da: DerivedArtifact[_ <: SourceArtifact])(implicit streamerFactory: StreamerFactory, transcodeSpec: TranscodeSpec) {

  private val sa = da.getSourceArtifact
  private val scanSpec = new ScanSpec(sa)
  private val workDir = sa.getFile.getParentFile
  private val derivedOverthereFile = OverthereUtils.getUniqueFolder(workDir, sa.getName).getFile(sa.getFile.getName)
  private val derivedFile = new File(derivedOverthereFile.getPath)

  private val placeholders = da.getPlaceholders.asScala.toMap
  private val byteBuffer = new Array[Byte](IOUtils.DEFAULT_BUFFER_SIZE)
  private val charBuffer = new Array[Char](IOUtils.DEFAULT_BUFFER_SIZE)

  def getTranscodeSpec : TranscodeSpec = transcodeSpec

  def enrichArtifact(): Unit = {
    if (sa == null) {
      da.setFile(null)
    } else {
      da.setFile(createDerivedFile())
    }
  }

  def createDerivedFile(): OverthereFile = {
    val sourceFile = sa.getFile
    logger.debug(s"Going to unpack $sourceFile into $derivedFile")
    if (!transcodeSpec.isDestinationHostRequireTranscodeToEbcdic && !shouldReplaceArtifactPlaceholders(da)) {
      logger.info(s"Not enriching $sa - placeholders not applicable, turned off, or none found")
      return LocalFile.valueOf(
        new PlaceholdersUtil.SourceArtifactUtil(sa).toLocalFile)
    }
    logger.debug(s"Going to replace placeholders for $da in ${derivedFile.getPath} before copying")
    replacePlaceholdersInArtifact()
    logger.info("Placeholder replacement done")
    derivedOverthereFile
  }

  private def copyStreamToFile(entry: StreamEntry, target: File): Unit = {
    Using.resources(new FileOutputStream(target), entry.getInputStream) { (w, is) =>
      copyBytes(is, w, byteBuffer)
    }
  }

  private def isValidArchive(is: InputStream, name: String) : Boolean = {
    val mappedExt = streamerFactory.getArchiveType(name)
    val resettableIs = ArtifactIOUtils.getResettableInputStream(is)
    archiveTypeMatchesExtension(resettableIs, mappedExt, name)
  }

  private[this] def replacePlaceholdersInArtifact(): Unit = {
    sa match {
      case fa: FolderArtifact =>
          logger.info(s"Going to replace placeholders in artifact $fa as in a folder artifact")
          replacePlaceholdersFileOrFolderArtifact()
      case _ if streamerFactory.hasArchiveExtension(sa.getFile.getName) =>
        logger.info(s"Going to replace placeholders in artifact $sa as in an archive artifact")
        replacePlaceholdersArchiveArtifact()

      case _ =>
        logger.info(s"Going to replace placeholders in artifact $sa from stream")
        replacePlaceholdersFileOrFolderArtifact()
    }
  }

  private[this] def replacePlaceholdersFileOrFolderArtifact(): Unit = {
    streamerFactory.streamer(sa.toLocalFile).stream().foreach { entry =>
      logger.debug(s"Replacing placeholders in ${entry.getPath}")
      doReplacePlaceholders(entry)
    }
  }

  private[this] def replacePlaceholdersArchiveArtifact(): Unit = {
    withArchiveOutputStream(derivedFile, os => {
      streamerFactory.streamer(sa.toLocalFile).stream().foreach {
        case e: XLArchiveEntry =>
          logger.debug(s"$sa: ${e.getName} is an archive entry file.")
          processEntry(e.getInputStream, os, e, placeholders)
      }
    })
  }

  private def doReplacePlaceholders(entry: StreamEntry): Unit = {
    if (entry.isDirectory) {
      val fileEntry = new File(derivedFile, entry.getPath)
      Files.createDirectories(fileEntry.toPath)
      return
    }

    scanSpec.getProcessingType(entry) match {
      case DigestOnly =>
        logger.debug(s"transcode detect for doReplacePlaceholders:$da: Skipping ${entry.getName}")
        val posixAttrForEntry = getPosixFilePermission(entry)
        val target = createDerivedFileDirectory(entry)
        transcodeSpec.detectTranscodeMode(entry) match {
          case Skip =>
            logger.debug(s"transcode:$da: Skipping ${entry.getName}")
            copyStreamToFile(entry, target)
          case Ebcdic =>
            logger.debug(s"transcode:$da: ${entry.getName} considered as a text file excluded in placeholder but included to transcode")
            Using.resource(entry.getInputStream) { is =>
              doReplacePlaceholdersTextFile(is, entry, target)
            }
        }
        setPosixFilePermissions(posixAttrForEntry,entry,target)
      case ProcessArchive if sa.isInstanceOf[FolderArtifact] =>
        logger.debug(s"$da: Detected archive for ${entry.getName} in source Folder artifact $sa")
        val posixAttrForEntry = getPosixFilePermission(entry)
        val target = createDerivedFileDirectory(entry)
        withArchiveOutputStream(target, os => {
          if(isValidArchive(entry.getInputStream, entry.getName)) {
            Using.resource(entry.getInputStream) { eis =>
              streamerFactory.streamer(eis, entry.getName).stream().foreach {
                case e: XLArchiveEntry =>
                  Using(e.getInputStream) {
                    processEntry(_, os, e, placeholders)
                  }
              }
            }
          } else {
            logger.warn(s"File ${entry.getName} is not a valid archive file. Skipping placeholder replacement.")
            copyStreamToFile(entry, target)
          }
        })
        setPosixFilePermissions(posixAttrForEntry,entry,target)

      case ProcessArchive =>
        logger.debug(s"$da: Detected archive for ${entry.getName} in source artifact $sa")
        Using.resource(entry.getInputStream) { is =>
          withArchiveOutputStream(new File(derivedFile.getPath), os => {
            processEntry(is, os, entry.asInstanceOf[XLArchiveEntry], placeholders)
          })
        }

      case ProcessTextFile =>
        logger.debug(s"$da: ${entry.getName} is a text file.")
        val (posixAttrForEntry, targetPath) = if (sa.getFile.isDirectory) {
          (getPosixFilePermission(entry), createDerivedFileDirectory(entry))
        } else {
          (null, derivedFile)
        }
        Using.resource(entry.getInputStream) { is =>
          doReplacePlaceholdersTextFile(is, entry, targetPath)
        }
        if (sa.getFile.isDirectory) {
          setPosixFilePermissions(posixAttrForEntry,entry,targetPath)
        }
    }
  }

  private[this] def doReplacePlaceholdersTextFile(is: InputStream, entry: StreamEntry, targetPath:File): Unit = {
    val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
    val charset =  detectCharset(sa, resettableInputStream, entry)
    val reader = readWithCharset(resettableInputStream, charset)

	val wrappedWriter = if(transcodeSpec.isTranscodeEbcdicRequired(entry)) {
      new OutputStreamWriter(new FileOutputStream(targetPath), getCharSetForEBCDIC)
    } else {
      charset.map(cs => new OutputStreamWriter(new FileOutputStream(targetPath), cs.newEncoder()))
        .getOrElse(new OutputStreamWriter(new FileOutputStream(targetPath)))
    }

    Using.Manager { use =>
      if(transcodeSpec.isDestinationHostRequireTranscodeToEbcdic && placeholders.isEmpty) {
        logger.debug(s"transcode doReplacePlaceholdersTextFile:$da: Skipping ${entry.getName} as it is empty placeholder and zos connection")
        copyChars(use(reader), use(wrappedWriter), charBuffer)
      } else {
        val readerForReplacePlaceHolder = scanSpec.mustacher.newReader(reader, placeholders)
        copyChars(use(readerForReplacePlaceHolder), use(wrappedWriter), charBuffer)
      }
    }
  }

  private def createDerivedFileDirectory(entry: StreamEntry): File = {
    val fileEntry = new File(derivedFile, entry.getPath)
    Files.createDirectories(fileEntry.getParentFile.toPath)
    fileEntry
  }

  private def getPosixFilePermission(entry: StreamEntry): PosixFileAttributeView = {
    entry match {
      case fileEntry: FileEntry if SystemUtils.IS_OS_LINUX || SystemUtils.IS_OS_MAC =>
        Files.getFileAttributeView(fileEntry.file.toPath, classOf[PosixFileAttributeView])
      case _ => null
    }
  }

  private def setPosixFilePermissions(posixView: PosixFileAttributeView, entry: StreamEntry, targetFile:File): Unit = {
    entry match {
       case fileEntry: FileEntry if (SystemUtils.IS_OS_LINUX || SystemUtils.IS_OS_MAC) && posixView != null  =>
         if (!targetFile.exists()) targetFile.createNewFile()
         val permissions = posixView.readAttributes.permissions
         Files.setPosixFilePermissions(targetFile.toPath, permissions)
       case _ =>
    }
  }

  private[this] def processEntry(is: InputStream, os: ArchiveOutputStream[ArchiveEntry], entry: XLArchiveEntry,
                                 placeholders: Map[String, String]): Unit = {
    val entryName = entry.getName

    scanSpec.getProcessingType(entry) match {
      case ProcessArchive =>
        processArchive()
      case ProcessTextFile if !entry.isDirectory =>
        processTextFile()
      case _ =>
        logger.debug(s"transcode detect for processEntry:$da: Skipping ${entry.getName}")
        transcodeSpec.detectTranscodeMode(entry) match {
          case Skip =>
            logger.debug(s"transcode:$da: Skipping ${entry.getName}")
            processOtherFileOrFolder()
          case Ebcdic =>
            logger.debug(s"transcode:$da: ${entry.getName} considered as a text file excluded in placeholder but included to transcode")
            processTextFile()
        }

    }

    def processArchive(): Unit = {
      logger.debug(s"Replacing placeholders in $entryName for path ${entry.getPath} as in an archive artifact")
      val now = UUID.randomUUID()
      val workDirLocalFile = workDir.asInstanceOf[LocalFile].getFile
      val inFile = new File(workDirLocalFile, s"in-$now-$entryName")
      val outFile = new File(workDirLocalFile, s"out-$now-$entryName")
      Using.Manager { use =>
        IOUtils.copy(use(entry.getInputStream), use(new FileOutputStream(inFile)))
      }
      if (!isValidArchive(new FileInputStream(inFile), inFile.getName)) {
        logger.warn(s"File $entryName is not a valid archive file. Skipping placeholder replacement.")
        Using.Manager { use =>
          IOUtils.copy(use(new FileInputStream(inFile)), use(new FileOutputStream(outFile)))
        }
      } else {
        withArchiveOutputStream(outFile, temporaryArchiveOutputStream => {
          val streamer: Streamer = streamerFactory.streamer(inFile)
          streamer.stream().foreach {
            case inner: XLArchiveEntry =>
              logger.debug(s"Processing entry [${inner.getName}] with path [${inner.getPath}] for parent entry [$entryName]")
              Using(inner.getInputStream) {
                processEntry(_, temporaryArchiveOutputStream, inner, placeholders)
              }
          }
        })
      }
      logger.debug(s"Processed entry [$entryName]")
      val ze = entry match {
        case e: JarArchivedEntry => new JarArchiveEntry(e.ze)
        case e: JarArchivedZipFileEntry => new JarArchiveEntry(e.ze)
        case e: ZipArchivedEntry => new ZipArchiveEntry(e.ze)
        case e: ZipArchivedZipFileEntry => new ZipArchiveEntry(e.ze)
        case e: ArchivedEntry => new TarArchiveEntry(outFile, e.ze.getName)
        case e => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        Using(new FileInputStream(outFile)) { is =>
          copyBytes(is, os, byteBuffer)
        } match {
          case Success(_) =>
            logger.debug(s"Copied data to the parent archive for entry [$entryName]")
            inFile.delete()
            outFile.delete()
            logger.debug("Deleted temporary data")
          case Failure(err) =>
            logger.error("Could not copy data from temporary file")
            logger.error(err.getMessage, err)
        }
      })
    }

    def processTextFile(): Unit = {
      logger.debug(s"Replacing placeholders in $entryName for path ${entry.getPath}")
      val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
      val charset = detectCharset(sa, resettableInputStream, entry)
      val reader = readWithCharset(resettableInputStream, charset)
      val mustacher = scanSpec.mustacher

      val (ze, convertedInputStream, processed) = entry match {
        case e: JarArchivedEntry =>
          (new JarArchiveEntry(e.ze), resettableInputStream, false)
        case e: JarArchivedZipFileEntry =>
          (new JarArchiveEntry(e.ze), resettableInputStream, false)
        case e: ZipArchivedEntry =>
          (new ZipArchiveEntry(e.ze), resettableInputStream, false)
        case e: ZipArchivedZipFileEntry =>
          (new ZipArchiveEntry(e.ze), resettableInputStream, false)
        case e: ArchivedEntry =>
          val tae = createTarArchiveEntry(e)
          if(transcodeSpec.isDestinationHostRequireTranscodeToEbcdic && placeholders.isEmpty) {
            val buf = Using(reader) {
              IOUtils.toByteArray(_, charset.getOrElse(defaultCharset()))
            }.get
            tae.setSize(buf.size)
            (tae, new ByteArrayInputStream(buf), true)
          } else {
            val readerWithPlaceholder = mustacher.newReader(reader, placeholders)
            val buf = Using(readerWithPlaceholder) {
              IOUtils.toByteArray(_, charset.getOrElse(defaultCharset()))
            }.get
            tae.setSize(buf.size)
            (tae, new ByteArrayInputStream(buf), true)
          }
        case e =>
          (e.ze, resettableInputStream, false)
      }

      withOpenArchiveEntry(os, ze, () => {
        val convertedReader = readWithCharset(convertedInputStream, charset)
        val wrappedReader = if (processed) {
          convertedReader
        } else {
          if(transcodeSpec.isDestinationHostRequireTranscodeToEbcdic && placeholders.isEmpty) {
            convertedReader
          } else {
            mustacher.newReader(convertedReader, placeholders)
          }
        }
        Using(wrappedReader) { reader =>
          if(transcodeSpec.isTranscodeEbcdicRequired(entry)) {
            copyCharsAndTranscode(reader, os, getCharSetForEBCDIC,charBuffer)
          } else {
            copyChars(reader, os, charset, charBuffer)
          }
        }
      })
    }

    def createTarArchiveEntry(ae: ArchivedEntry): TarArchiveEntry = {
      val tae = new TarArchiveEntry(ae.ze.getName)
      if (ae.ze.isInstanceOf[TarArchiveEntry]) {
        val unixMode = ae.ze.asInstanceOf[TarArchiveEntry].getMode
        if (unixMode != 0)
          tae.setMode(unixMode)
      }
      tae
    }

    def processOtherFileOrFolder(): Unit = { // other files and folders, we cannot skip folders because those can be used
      logger.debug(s"Processing $entryName for path ${entry.getPath}")
      val ze = entry match {
        case e: JarArchivedEntry => new JarArchiveEntry(e.ze)
        case e: JarArchivedZipFileEntry => new JarArchiveEntry(e.ze)
        case e: ZipArchivedEntry => new ZipArchiveEntry(e.ze)
        case e: ZipArchivedZipFileEntry => new ZipArchiveEntry(e.ze)
        case e: ArchivedEntry => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        copyBytes(is, os, byteBuffer)
      })
    }
  }

}
