/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.io

import java.io._
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.util.TryWith
import com.xebialabs.xldeploy.packager.PackagerConfig
import com.xebialabs.xldeploy.packager.io.StreamerFactory.logger
import com.xebialabs.xldeploy.packager.io.SupportedArchiveExtensions._
import grizzled.slf4j.Logger
import org.apache.commons.compress.archivers.{ArchiveEntry, ArchiveOutputStream}
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream
import org.apache.commons.compress.compressors.bzip2.{BZip2CompressorInputStream, BZip2CompressorOutputStream}
import org.apache.commons.compress.compressors.gzip.{GzipCompressorInputStream, GzipCompressorOutputStream}
import org.slf4j.LoggerFactory

object StreamerFactory {

  private[StreamerFactory] val logger = new Logger(LoggerFactory.getLogger(classOf[StreamerFactory]))

  def defaultMappings(): StreamerFactory = forConfig(ConfigFactory.defaultReference())

  def forConfig(config: Config): StreamerFactory = {
    val packagerConfig = new PackagerConfig(config)
    logger.debug(s"stream:2G:Creating StreamerFactory with PackagerConfig threshold: ${packagerConfig.inMemoryThreshold}")
    val factory = new StreamerFactory(packagerConfig.archiveExtensionMappings, packagerConfig.zipEncoding,
      packagerConfig.jarEncoding, packagerConfig.inMemoryThreshold)
    factory
  }
}

class StreamerFactory(archiveMappings: Map[String, String], zipEncoding: String, jarEncoding: String, 
                     inMemoryThreshold: Long = PackagerConfig.MAX_THRESHOLD_BYTES) {

  private[this] def getArchiveEntryStreamer(file: File): Streamer = {
    val mappedExt = getArchiveType(file.getName)

    val extOK = TryWith(ArtifactIOUtils.getResettableInputStream(new FileInputStream(file))) { resettableIs =>
      archiveTypeMatchesExtension(resettableIs, mappedExt, file.getName)
    }.get

    mappedExt match {
      case _ if !extOK => new EmptyStreamer
      case ZIP => new ZipFileArchiveStreamer(file, zipEncoding)
      case JAR => new JarFileArchiveStreamer(file, jarEncoding)
      case TAR => new ArchiveStreamer(file, is => is)
      case TARGZ => new ArchiveStreamer(file, is => new GzipCompressorInputStream(is))
      case TARBZ2 => new ArchiveStreamer(file, is => new BZip2CompressorInputStream(is))
    }
  }

  // used for scanning
  private[io] def getArchiveEntryStreamStreamer(is: InputStream, name: String, useStreamingMode: Boolean = false): Streamer = {
    val mappedExt = getArchiveType(name)

    val resettableIs = ArtifactIOUtils.getResettableInputStream(is)
    val extOK = archiveTypeMatchesExtension(resettableIs, mappedExt, name)

    mappedExt match {
      case _ if !extOK => new EmptyStreamer
      case ZIP if useStreamingMode =>
        logger.debug(s"stream:2G:Creating ZipArchiveStreamStreamer (streaming) for ZIP: $name")
        new ZipArchiveStreamStreamer(resettableIs, zipEncoding)
      case ZIP =>
        logger.debug(s"stream:2G:Creating ZipFileArchiveStreamStreamer (standard) for ZIP: $name")
        new ZipFileArchiveStreamStreamer(resettableIs, zipEncoding)
      case JAR if useStreamingMode =>
        logger.debug(s"stream:2G:Creating ZipArchiveStreamStreamer (streaming) for JAR: $name")
        new ZipArchiveStreamStreamer(resettableIs, jarEncoding)
      case JAR =>
        logger.debug(s"stream:2G:Creating JarArchiveStreamStreamer (standard) for JAR: $name")
        new JarArchiveStreamStreamer(resettableIs, jarEncoding)
      case TAR =>
        new ArchiveStreamStreamer(resettableIs, is => is)
      case TARGZ =>
        new ArchiveStreamStreamer(resettableIs, is => new GzipCompressorInputStream(is))
      case TARBZ2 =>
        new ArchiveStreamStreamer(resettableIs, is => new BZip2CompressorInputStream(is))
    }
  }

  def getArchiveOutputStream(target: File): ArchiveOutputStream[? <: ArchiveEntry] = {
    getArchiveType(target.getName) match {
      case ZIP => new ZipArchiveOutputStream(target)
      case JAR => new ZipArchiveOutputStream(target) // JarArchiveOutputStream doesn't have this constructor. It is required for correct function.
      case TAR => getTarArchiveOutputStream(target, os => os)
      case TARGZ => getTarArchiveOutputStream(target, os => new GzipCompressorOutputStream(os))
      case TARBZ2 => getTarArchiveOutputStream(target, os => new BZip2CompressorOutputStream(os))

      case mappedExt =>
        val originExt = archiveMappings.find(p = am => target.getName.endsWith('.' + am._1) || target.getName.endsWith('.' + am._1.replace('-', '.')) ).get._1
        throw UnsupportedArchiveExtensionException(
          s"${target.getName} with extension $originExt -> $mappedExt is not a supported archive output type")
    }
  }

  private[this] def getTarArchiveOutputStream(target: File, compressor: OutputStream => OutputStream): TarArchiveOutputStream = {
    val wrapped = compressor(new FileOutputStream(target)) // TODO will these be closed properly?
    val os = new TarArchiveOutputStream(wrapped)
    os.setLongFileMode(TarArchiveOutputStream.LONGFILE_POSIX)
    os.setBigNumberMode(TarArchiveOutputStream.BIGNUMBER_POSIX)
    os
  }

  def hasArchiveExtension(name: String): Boolean = {
    archiveMappings.exists(am => (name.endsWith('.' + am._1) || name.endsWith('.' + am._1.replace('-', '.'))))
  }

  def streamer(is: InputStream, name: String, isArchive: Boolean = false, fileSize: Long = -1L): Streamer = {
    val shouldUseStreaming = fileSize > 0 && inMemoryThreshold > 0 && fileSize >= inMemoryThreshold
    logger.debug(s"stream:2G:File size: $fileSize, Threshold: $inMemoryThreshold, shouldUseStreaming: $shouldUseStreaming")
    if (hasArchiveExtension(name)) {
        getArchiveEntryStreamStreamer(is, name, shouldUseStreaming)
    } else if (isArchive) {
      if (shouldUseStreaming) {
        logger.debug(s"stream:2G:Large file mode enabled for archive without extension - using ZipArchiveStreamStreamer")
        new ZipArchiveStreamStreamer(ArtifactIOUtils.getResettableInputStream(is), zipEncoding)
      } else {
        new ZipFileArchiveStreamStreamer(ArtifactIOUtils.getResettableInputStream(is), zipEncoding)
      }
    } else {
      new FileStreamStreamer(is, name)
    }
  }

  def streamer(file: File): Streamer = {
    if (file.isDirectory) {
      new DirectoryStreamer(file)
    } else if (hasArchiveExtension(file.getName)) {
      getArchiveEntryStreamer(file)
    } else {
      new FileStreamer(file)
    }
  }

  def getArchiveType(name: String): String = {
    archiveMappings.find(am => name.endsWith('.' + am._1) || name.endsWith('.' + am._1.replace('-', '.'))) match {
      case None =>
        throw UnsupportedArchiveExtensionException(s"$name is not a supported archive")
      case Some((originExt, mappedExt)) =>
        logger.debug(s"Detected mapped archive extension $originExt -> $mappedExt for $name")
        mappedExt
    }
  }
}
