/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.placeholders

import com.xebialabs.deployit.plugin.api.udm.artifact.{FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.util.TryWith
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.Mustacher
import com.xebialabs.xldeploy.packager.io.ArtifactIOUtils.{detectCharset, readWithCharset}
import com.xebialabs.xldeploy.packager.io.{ArtifactIOUtils, DevNull, StreamEntry, StreamerFactory}
import com.xebialabs.xldeploy.packager.placeholders.PlaceholdersUtil._
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner._
import grizzled.slf4j.Logger
import org.apache.commons.codec.binary.Hex

import java.io.{File, InputStream, OutputStreamWriter, Reader}
import java.nio.charset.StandardCharsets
import java.security.{DigestInputStream, MessageDigest}
import scala.jdk.CollectionConverters._

final case class StreamFile(name: String, inputStream: InputStream)

object SourceArtifactScanner {

  private[SourceArtifactScanner] val logger = new Logger(org.slf4j.LoggerFactory.getLogger(classOf[SourceArtifactScanner]))

  private[SourceArtifactScanner] val devNull = new DevNull

  private[SourceArtifactScanner] val devNullWriter = new OutputStreamWriter(devNull)
}

class SourceArtifactScanner(streamerFactory: StreamerFactory) {

  def enrichArtifact(sa: SourceArtifact, streamFile: Option[StreamFile], messageDigest: () => MessageDigest): Unit = {
    scanAndCalculateDigest(sa, streamFile, messageDigest)
  }

  def scanAndCalculateDigest(sa: SourceArtifact, streamFile: Option[StreamFile], messageDigest: () => MessageDigest,
                             isRescan: Boolean = false): Unit = {
    if (!sa.shouldScanPlaceholders() && sa.hasCheckSum) {
      logger.info(s"Not enriching $sa as no placeholders should be scanned and checksum is present")
      return
    }

    val digest: Option[MessageDigest] = if (sa.shouldGenerateChecksum(isRescan)) Some(messageDigest()) else None
    val mustacher = Mustacher(sa.mustacheDelimiters())

    calculatePlaceholdersAndDigest(sa, streamFile, mustacher, digest)

    sa.setPlaceholders(mustacher.placeholders.asJava)
    digest.foreach(d => sa.setProperty(SourceArtifact.CHECKSUM_PROPERTY_NAME, Hex.encodeHexString(d.digest())))
  }

  private[this] def calculatePlaceholdersAndDigest(sa: SourceArtifact, streamFile: Option[StreamFile], mustacher: Mustacher,
                                                   optDigest: Option[MessageDigest]): Unit = {
    sa match {
      case fa: FolderArtifact =>
        logger.info(s"Going to scan artifact $fa as a folder artifact")
        scanFolderArtifact(fa, streamFile, optDigest, mustacher)
      case _ if streamFile.isEmpty =>
        logger.info(s"Going to scan artifact $sa")
        TryWith[InputStream, Unit](sa.getFile.getInputStream) { fis =>
          scanFileOrArchiveArtifact(sa, StreamFile(sa.getFile.getName, fis), optDigest, mustacher)
        }
      case _ =>
        logger.info(s"Going to scan artifact $sa from stream")
        scanFileOrArchiveArtifact(sa, streamFile.get, optDigest, mustacher)
    }
  }

  private[this] def scanFolderArtifact(fa: FolderArtifact, streamFile: Option[StreamFile], digest: Option[MessageDigest],
                                       mustacher: Mustacher): Unit = {
    val streamer = streamFile match {
      case Some(x) => streamerFactory.streamer(x.inputStream, x.name)
      case None => streamerFactory.streamer(fa.toLocalFile)
    }
    streamer.stream().foreach { entry =>
      digest.foreach(d => updateDigestWithFilename(entry, d))
      doScan(fa, entry, mustacher, is => digest.map(new DigestInputStream(is, _)).getOrElse(is))
    }
  }

  private[this] def scanFileOrArchiveArtifact(sa: SourceArtifact, streamFile: StreamFile,
                                              digest: Option[MessageDigest], mustacher: Mustacher): Unit = {
    val inputStream = digest.map(new DigestInputStream(streamFile.inputStream, _)).getOrElse(streamFile.inputStream)
    if (!sa.shouldScanPlaceholders()) {
      logger.debug(s"Artifact [$sa] has disabled placeholder scanning")
    } else {
      streamerFactory.streamer(inputStream, streamFile.name).stream().foreach { entry =>
        doScan(sa, entry, mustacher, identity[InputStream])
      }
    }
    // Ensure that we empty the stream so that all bytes are read, testcases seem to prove this is not needed,
    // But we don't want to accidentally calculate wrong checksum.
    OverthereUtils.write(inputStream, devNull)
  }

  private[this] def updateDigestWithFilename(entry: StreamEntry, digest: MessageDigest): Unit = {
    var digestPath = entry.getPath + (if (entry.isDirectory) File.separator else "")
    digestPath = digestPath.replace("\\", "/")
    logger.trace(s"scanFolderArtifact: update digest with path [$digestPath]")
    digest.update(digestPath.getBytes(StandardCharsets.UTF_8))
  }

  private[this] def doScan(sa: SourceArtifact, entry: StreamEntry, mustacher: Mustacher,
                           transform: InputStream => InputStream): Unit = {

    if (entry.isDirectory) { return }

    TryWith(entry.getInputStream) { inputStream =>
      val is = transform(inputStream)

      sa.getScanTypeForEntry(entry)(streamerFactory) match {
        case DigestOnly =>
          logger.debug(s"$sa: Skipping ${entry.getName}")
          OverthereUtils.write(is, devNull) // Just write the bytes and calculate the digest

        case ProcessArchive =>
          logger.debug(s"$sa: Detected archive for ${entry.getName}")
          streamerFactory.streamer(is, entry.getName).stream().foreach { e2 =>
            doScan(sa, e2, mustacher, identity[InputStream])
          }
          // Ensure that we empty the stream so that all bytes are read, so that we don't accidentally calculate wrong checksum.
          OverthereUtils.write(is, devNull) // Finish the archive

        case ProcessTextFile =>
          logger.debug(s"$sa: ${entry.getName} is a text file.")
          val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
          val charset = detectCharset(sa, resettableInputStream, entry)
          val reader: Reader = readWithCharset(resettableInputStream, charset)
          TryWith(mustacher.newReader(reader)) { r =>
            OverthereUtils.write(r, devNullWriter) // Scan the file for placeholders using the Mustacher
          }
      }
    }
  }
}
