/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xldeploy.packager.placeholders

import java.io._
import java.nio.charset.Charset.defaultCharset

import com.xebialabs.deployit.plugin.api.udm.artifact.{DerivedArtifact, FolderArtifact, SourceArtifact}
import com.xebialabs.deployit.util.TryWith
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xldeploy.packager.io.ArtifactIOUtils._
import com.xebialabs.xldeploy.packager.io._
import com.xebialabs.xldeploy.packager.placeholders.DerivedArtifactEnricher._
import com.xebialabs.xldeploy.packager.placeholders.PlaceholdersUtil._
import org.apache.commons.compress.archivers.ArchiveOutputStream
import org.apache.commons.compress.archivers.jar.JarArchiveEntry
import org.apache.commons.compress.archivers.tar.TarArchiveEntry
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry
import org.apache.commons.io.IOUtils
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._
import scala.language.postfixOps
import scala.util.{Failure, Success}

object DerivedArtifactEnricher {
  private[DerivedArtifactEnricher] val logger: Logger = LoggerFactory.getLogger(classOf[DerivedArtifactEnricher])

  private[DerivedArtifactEnricher] def shouldReplaceArtifactPlaceholders(da: DerivedArtifact[_ <: SourceArtifact]): Boolean = {
    da.getSourceArtifact.shouldScanPlaceholders() && da.getPlaceholders != null && da.getPlaceholders.size() > 0
  }
}

class DerivedArtifactEnricher(da: DerivedArtifact[_ <: SourceArtifact])(implicit streamerFactory: StreamerFactory) {

  private val sa = da.getSourceArtifact
  private val scanSpec = new ScanSpec(sa)
  private val workDir = sa.getFile.getParentFile
  private val derivedOverthereFile = OverthereUtils.getUniqueFolder(workDir, sa.getName).getFile(sa.getFile.getName)
  private val derivedFile = new File(derivedOverthereFile.getPath)

  def enrichArtifact(): Unit = {
    if (sa == null) {
      da.setFile(null)
    } else {
      da.setFile(createDerivedFile())
    }
  }

  def createDerivedFile(): OverthereFile = {
    val sourceFile = sa.getFile

    logger.debug(s"Going to unpack $sourceFile into $derivedFile")
    sourceFile.copyTo(derivedOverthereFile)
    logger.debug(s"Going to replace placeholders for $da in ${derivedFile.getPath}")
    replacePlaceholdersInArtifact()
    logger.debug("Placeholder replacement done")

    derivedOverthereFile
  }

  private[this] def replacePlaceholdersInArtifact(): Unit = {
    if (!shouldReplaceArtifactPlaceholders(da)) {
      logger.info(s"Not enriching $sa - placeholders not applicable, turned off, or none found")
      return
    }

    sa match {
      case fa: FolderArtifact =>
        logger.info(s"Going to replace placeholders in artifact $fa as in a folder artifact")
        replacePlaceholdersFileOrFolderArtifact()

      case _ if streamerFactory.hasArchiveExtension(sa.getFile.getName) =>
        logger.info(s"Going to replace placeholders in artifact $sa as in an archive artifact")
        replacePlaceholdersArchiveArtifact()

      case _ =>
        logger.info(s"Going to replace placeholders in artifact $sa from stream")
        replacePlaceholdersFileOrFolderArtifact()
    }
  }

  private[this] def replacePlaceholdersFileOrFolderArtifact(): Unit = {
    streamerFactory.streamer(sa.toLocalFile).stream().foreach { entry =>
      logger.debug(s"Replacing placeholders in ${entry.getPath}")
      doReplacePlaceholders(entry)
    }
  }


  private[this] def replacePlaceholdersArchiveArtifact(): Unit = {
    val parentFile = derivedFile.getParentFile
    withArchiveOutputStream(derivedFile, os => {
      streamerFactory.streamer(sa.toLocalFile).stream().foreach {
        case e: XLArchiveEntry =>
          logger.debug(s"$sa: ${e.getName} is an archive entry file.")
          processEntry(e.getInputStream, os, e, da.getPlaceholders.asScala.toMap)
      }
    })
  }

  private def doReplacePlaceholders(entry: StreamEntry): Unit = {
    if (entry.isDirectory) { return }

    scanSpec.getProcessingType(entry) match {
      case DigestOnly =>
        logger.debug(s"$da: Skipping ${entry.getName}")

      case ProcessArchive if sa.isInstanceOf[FolderArtifact] =>
        logger.debug(s"$da: Detected archive for ${entry.getName} in source Folder artifact $sa")
        val target = new File(derivedFile.getPath, entry.getPath)
        withArchiveOutputStream(target, os => {
          TryWith(entry.getInputStream) { eis =>
            streamerFactory.streamer(eis, entry.getName).stream().foreach {
              case e: XLArchiveEntry =>
                processEntry(e.getInputStream, os, e, da.getPlaceholders.asScala.toMap)
            }
          }
        })

      case ProcessArchive =>
        logger.debug(s"$da: Detected archive for ${entry.getName} in source artifact $sa")
        TryWith(entry.getInputStream) { is =>
          withArchiveOutputStream(new File(derivedFile.getPath), os => {
            processEntry(is, os, entry.asInstanceOf[XLArchiveEntry], da.getPlaceholders.asScala.toMap)
          })
        }

      case ProcessTextFile =>
        logger.debug(s"$da: ${entry.getName} is a text file.")
        TryWith(entry.getInputStream) { is =>
          doReplacePlaceholdersTextFile(is, entry)
        }
    }
  }

  private[this] def doReplacePlaceholdersTextFile(is: InputStream, entry: StreamEntry): Unit = {
    val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
    val charset = detectCharset(sa, resettableInputStream, entry)
    val reader = readWithCharset(resettableInputStream, charset)

    val targetPath = if (derivedFile.isDirectory) {
      new File(derivedFile, entry.getPath)
    } else {
      derivedFile
    }

    val wrappedWriter = charset.map(cs => new OutputStreamWriter(new FileOutputStream(targetPath), cs.newEncoder()))
      .getOrElse(new OutputStreamWriter(new FileOutputStream(targetPath)))

    TryWith(scanSpec.mustacher.newReader(reader, da.getPlaceholders.asScala.toMap)) { r =>
      TryWith(wrappedWriter) { w =>
        OverthereUtils.write(r, w)
      }
    }
  }

  private[this] def processEntry(is: InputStream, os: ArchiveOutputStream, entry: XLArchiveEntry,
                                 placeholders: Map[String, String]): Unit = {
    val entryName = entry.getName

    scanSpec.getProcessingType(entry) match {
      case ProcessArchive =>
        processArchive()
      case ProcessTextFile if !entry.isDirectory =>
        processTextFile()
      case _ =>
        processOtherFileOrFolder()
    }

    def processArchive(): Unit = {
      logger.debug(s"Replacing placeholders in $entryName for path ${entry.getPath} as in an archive artifact")
      val now = System.currentTimeMillis()
      val workDirLocalFile = workDir.asInstanceOf[LocalFile].getFile
      val inFile = new File(workDirLocalFile, s"in-$now-$entryName")
      val outFile = new File(workDirLocalFile, s"out-$now-$entryName")
      TryWith(new FileOutputStream(inFile)) { IOUtils.copy(entry.getInputStream, _) }
      withArchiveOutputStream(outFile, temporaryArchiveOutputStream => {
        streamerFactory.streamer(inFile).stream().foreach {
          case inner: XLArchiveEntry =>
            logger.debug(s"Processing entry [${inner.getName}] with path [${inner.getPath}] for parent entry [$entryName]")
            processEntry(inner.getInputStream, temporaryArchiveOutputStream, inner, placeholders)
        }
      })
      logger.debug(s"Processed entry [$entryName]")
      val ze = entry match {
        case e: JarArchivedEntry => new JarArchiveEntry(e.ze)
        case e: ZipArchivedEntry => new ZipArchiveEntry(e.ze)
        case e: ArchivedEntry => new TarArchiveEntry(outFile, e.ze.getName)
        case e => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        TryWith(new FileInputStream(outFile)) { is =>
          OverthereUtils.write(is, os)
        } match {
          case Success(_) =>
            logger.debug(s"Copied data to the parent archive for entry [$entryName]")
            inFile.delete()
            outFile.delete()
            logger.debug("Deleted temporary data")
          case Failure(err) =>
            logger.error("Could not copy data from temporary file")
            logger.error(err.getMessage, err)
        }
      })
    }

    def processTextFile(): Unit = {
      logger.debug(s"Replacing placeholders in $entryName for path ${entry.getPath}")
      val resettableInputStream = ArtifactIOUtils.getResettableInputStream(is)
      val charset = detectCharset(sa, resettableInputStream, entry)
      val reader = readWithCharset(resettableInputStream, charset)
      val mustacher = scanSpec.mustacher

      val (ze, convertedInputStream, processed) = entry match {
        case e: JarArchivedEntry =>
          (new JarArchiveEntry(e.ze), resettableInputStream, false)
        case e: ZipArchivedEntry =>
          (new ZipArchiveEntry(e.ze), resettableInputStream, false)
        case e: ArchivedEntry =>
          val wrappedReader = mustacher.newReader(reader, placeholders)
          val tae = new TarArchiveEntry(e.ze.getName)
          val buf = IOUtils.toByteArray(wrappedReader, charset.getOrElse(defaultCharset()))
          tae.setSize(buf.size)
          (tae, new ByteArrayInputStream(buf), true)
        case e =>
          (e.ze, resettableInputStream, false)
      }

      withOpenArchiveEntry(os, ze, () => {
        val convertedReader = readWithCharset(convertedInputStream, charset)
        val wrappedReader = if (processed) {
          convertedReader
        } else {
          mustacher.newReader(convertedReader, placeholders)
        }
        copyBytes(wrappedReader, os, charset)
      })
    }

    def processOtherFileOrFolder(): Unit = { // other files and folders, we cannot skip folders because those can be used
      logger.debug(s"Processing $entryName for path ${entry.getPath}")
      val ze = entry match {
        case e: JarArchivedEntry => new JarArchiveEntry(e.ze)
        case e: ZipArchivedEntry => new ZipArchiveEntry(e.ze)
        case e: ArchivedEntry => e.ze
      }
      withOpenArchiveEntry(os, ze, () => {
        OverthereUtils.write(is, os)
      })
    }
  }

}
